/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.gcn.plinguacore.simulator.cellLike.stochastic;

import java.io.PrintStream;
import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcn.plinguacore.simulator.AbstractSimulator;
import org.gcn.plinguacore.util.GeneString;
import org.gcn.plinguacore.util.MultiSet;
import org.gcn.plinguacore.util.Pair;
import org.gcn.plinguacore.util.PlinguaCoreException;
import org.gcn.plinguacore.util.psystem.AlphabetObject;
import org.gcn.plinguacore.util.psystem.Psystem;
import org.gcn.plinguacore.util.psystem.cellLike.membrane.CellLikeMembrane;
import org.gcn.plinguacore.util.psystem.cellLike.membrane.CellLikeNoSkinMembrane;
import org.gcn.plinguacore.util.psystem.membrane.ChangeableMembrane;
import org.gcn.plinguacore.util.psystem.membrane.Membrane;
import org.gcn.plinguacore.util.psystem.membrane.MembraneStructure;
import org.gcn.plinguacore.util.psystem.rule.IRule;
import org.gcn.plinguacore.util.psystem.rule.IConstantRule;


/**

 *  @author Research Group on Natural Computing (http://www.gcn.us.es)
 *
 *
 * 
 *         General Simulator of stochastic P-systems. It uses the
 *         "Template Method Pattern". In step() we provide the basic algorithm
 *         to perform an step in the simulation, but particular details such as
 *         choosing the next reaction to be executed or update the rest of
 *         reactions are subject to particular implementations. Those methods
 *         are defined as abstract in this class.
 * 
 */
public abstract class StochasticSimulator extends AbstractSimulator {

	/**
	 * Generated Serial Version UID
	 */
	private static final long serialVersionUID = 1519762430296909262L;

	private double timeOut; // Time at which to end the simulation
	protected List<Reaction> reactions; // List of reactions. Each reaction
	// corresponds to one rule in the
	// P-system
	protected double t; // "time of Gillespie"
	private double T; // time of sampling
	private int numberOfSteps; // number of output time steps
	private Map<String, double[]> variables; // result of simulation


	/**
	 * Constructor. Initializes attributes and builds the list of reactions
	 * using the rules.
	 * 
	 * @param psystem
	 *            The P-System to be simulated
	 */
	public StochasticSimulator(Psystem psystem) {
		super(psystem);
		currentConfig = psystem.getFirstConfiguration();
		t = 0;
		T = 0;
		reactions = new LinkedList<Reaction>();

		// Build the list of reactions. Associate every rule of the psystem with
		// a reaction of the chemical system
		Iterator<IRule> it = psystem.getRules().iterator();
		while (it.hasNext()) {
			IRule next = it.next();
			Reaction r = new Reaction(next);
			reactions.add(r);
		}

		// Default values for number of steps and time out
		timeOut = 50;
		numberOfSteps = 50;

		// This attribute holds the results of the simulation
		variables = new HashMap<String, double[]>();

	}

	/**
	 * Sets the time of the stochastic simulation to zero and initializes the
	 * list of reactions, that is, a_i is recalculated for every reaction in the
	 * system. Reset also the list of results.
	 */
	@Override
	public void reset() {
		super.reset();
		t = 0;
		T = 0;
		reactions = new LinkedList<Reaction>();

		// Every rule of the system is associated with a reaction
		Iterator<IRule> it = getPsystem().getRules().iterator();
		while (it.hasNext()) {
			IRule next = it.next();
			Reaction r = new Reaction(next);
			reactions.add(r);
		}

		variables = new HashMap<String, double[]>();
	}

	/**
	 * Performs a step of simulation. In the case of the Gillespie algorithm,
	 * that means that only one reaction takes place
	 */
	@Override
	protected boolean specificStep() throws PlinguaCoreException {
		// Beginning of simulation. Initialize variables and save initial state
		if (t == 0) {
			Iterator<AlphabetObject> it2 = getPsystem().alphabetIterator();
			while (it2.hasNext()) {
				String a = it2.next().toString();
				variables.put(a, new double[numberOfSteps + 1]);
			}

			saveCurrent();
		}

		// next "point" to save content: T + size of one step
		T += timeOut / numberOfSteps;

		// Check if the simulation is finished, because we have reached the
		// timeout
		if (T > timeOut) {
			return false;
		}

		// Not leave "holes" xD
		while (t > T && T < timeOut) {
			T += timeOut / numberOfSteps;
			saveCurrent();
		}
		// continue executing reactions till the next sampling instant is
		// reached
		// one step comprises the time between a sampling instant and the next
		// one
		while (t <= T) {

			// next reaction to be executed
			Reaction r_mu = getNextReaction();

			// The next reaction cannot be executed. The simulation is
			// terminated
			if (r_mu == null || r_mu.getA() <= 0)
				return false;

			// Save current state, to keep the state of the system in T,
			// before executing r_mu
			saveCurrent();

			// execute r_mu
			// that produces a change in the number of molecules in the system
			r_mu.execute();

			// update the "real" time of the system
			t = getNextTime();

			// update system to reflect the changes caused by the execution of
			// r_mu
			updateSystem();

		}

		// If we are at this point, T<timeOut and r_mu was executed
		// successfully.
		// Continue with the simulation
		return true;

	}

	/**
	 * @return Next reaction to be executed. The criteria to choose such
	 *         reaction depends on which algorithm is used (Gillespie's classic
	 *         direct algorithm, Gillespie's First Reaction method, Efficient
	 *         algorithm based on priority queues...)
	 */
	public abstract Reaction getNextReaction();

	/**
	 * @return Next real time of the system. Also depends on the algorithm used
	 *         to choose the next reaction, since that time is calculated
	 *         according to the tau of the reaction executed.
	 * @see getNextReaction()
	 */
	public abstract double getNextTime();

	/**
	 * Updates the stochastic attributes of the reactions in the system. Also
	 * depends on which algorithm we are using. For example, with the
	 * Gillespie's algorithms, all constants of all reactions are recalculated.
	 * With the efficient algorithm, a dependency graph is used to recalculate
	 * only the reactions affected.
	 */
	public abstract void updateSystem();

	/**
	 * @return A map containing the result of the simulation. The elements of
	 *         this maps are pairs of the form <species, [value_at(t0),
	 *         value_at(t1),...,value_at(timeout)]> The size of the list of
	 *         values is exactly the number of steps + 1
	 */
	public Map<String, double[]> getResults() {
		return variables;
	}

	/**
	 * Writes the results of a simulation (stored in "variables") in CSV format
	 * (comma separated values), but using the separator provided as parameter:
	 * Time,Var1,Var2... 0,90,87,0,.. .... Uses the set PrintStream for the
	 * simulator, given by getInfoChannel()
	 * 
	 * @param sep
	 *            Character or string separator between columns
	 */
	public void writeResults(String sep) {

		PrintStream p = getInfoChannel();

		// Headers
		p.print("Time");
		Iterator<String> it = variables.keySet().iterator();
		while (it.hasNext()) {
			String v = it.next();
			p.print(sep + v);
		}

		// Content
		for (int i = 0; i <= numberOfSteps; i++) {
			p.println();
			p.format("%.4f", i * timeOut / numberOfSteps);
			it = variables.keySet().iterator();
			while (it.hasNext()) {
				String v = it.next();
				p.print(sep + (int) variables.get(v)[i]);
			}
		}

	}

	/**
	 * Sets the timeOut for the Gillespie simulation The default timeOut is 50
	 * 
	 * @param timeOut
	 *            the timeOut to set
	 */
	public void setTimeOut(double timeOut) {
		this.timeOut = timeOut;
	}

	/**
	 * Sets the number of output time steps Default is 50
	 * 
	 * @param numberOfSteps
	 *            the numberOfSteps to set
	 */
	public void setNumberOfSteps(int numberOfSteps) {
		this.numberOfSteps = numberOfSteps;
	}

	/*
	 * Save the current state of the system, that is, in T, store the value of
	 * each species s in variables(s)[T]
	 */
	private void saveCurrent() {

		Map<String, Integer> content = currentGlobalContent();
		Iterator<String> its = content.keySet().iterator();

		while (its.hasNext()) {
			String obj = its.next();
			double[] values;
			if (variables.containsKey(obj))
				values = variables.get(obj);

			else
				values = new double[numberOfSteps + 1];

			values[(int) (T * numberOfSteps / timeOut)] = content.get(obj);
			variables.put(obj, values);
		}

	}

	/*
	 * Returns a map containing all the elements of the system, together with
	 * their multiplicity.
	 */
	private Map<String, Integer> currentGlobalContent() {
		Map<String, Integer> res = new HashMap<String, Integer>();

		// Go through the membranes in the P-system, counting the elements
		Iterator<? extends Membrane> it = currentConfig.getMembraneStructure().getAllMembranes()
				.iterator();
		while (it.hasNext()) {
			MultiSet<String> m = it.next().getMultiSet();
			Iterator<String> itm = m.iterator();
			while (itm.hasNext()) {
				String s = itm.next();
				if (!res.containsKey(s))
					res.put(s, 1);
				else
					res.put(s, res.get(s).intValue() + 1);

			}
		}

		return res;
	}

	/**
	 * Class Reaction
	 * 
	 * @author Rosa Gutiérrez Escudero This class represents a reaction (also
	 *         called "reaction channel") in a biochemical system. One reaction
	 *         in the biochemical system is directly associated with a rule in
	 *         the p-system. The general form of a biochemical reaction is:
	 *         a[b]'l --> c[d]'l where a,b,c,d are all optional and can be
	 *         multisets or strings. a and b represent the reactants, and c and
	 *         d the products
	 * 
	 *         This class implements also Comparable<Reaction>, according to the
	 *         time at which the reactions are scheduled to execute.
	 * 
	 *         We identify a Reaction in a unique way using the pair
	 *         (membrane(label), index).
	 */
	public final class Reaction implements Comparable<Reaction>, Serializable {

		/**
		 * 
		 */
		private static final long serialVersionUID = 5501749731967766458L;
		private double a; // a*dt is the probability that this reaction will
		// occur in the next infinitesimal time interval (t, t+dt)
		private double c; // Stochastic constant
		private double tau; // time at which the reaction is scheduled
		private int index; // id of this reaction. It has to be unique for this
		// membrane
		private int globalIndex; // global id for this membrane. Unique in the
		// whole system
		private IRule rule; // rule associated with this reaction
		private CellLikeMembrane membrane; // Membrane in the P-system of the
		// left side of the rule
		private CellLikeMembrane parentMembrane; // Parent membrane of the
		// previous one

		private double t_1; // this attribute is used to keep the time when
		// a first becomes 0 (if this happens)
		private double a_before_zero; // and this one is used to keep the last
		// value of a before becoming 0
		private Map<String, Long> reactants; // Set of reactants inside of
		// the left membrane
		private Map<String, Long> outerReactants; // Set of reactants outside
		// of the left membrane
		private Set<String> products; // Set of products inside of right
		// membrane
		private Set<String> outerProducts; // Set of products outside of the

		// right membrane

		
			public Reaction(IRule rule) {
			this.rule = rule;
			this.c = ((IConstantRule) rule).getConstant();
			this.a_before_zero = 0;
			this.t_1 = 0;

			// Search the membrane associated with this rule
			Iterator<? extends Membrane> it = currentConfig.getMembraneStructure().getAllMembranes()
					.iterator();
			boolean end = false;

			while (it.hasNext() && !end) {
				CellLikeMembrane m = (CellLikeMembrane) it.next();
				if (m.getLabel().equalsIgnoreCase(
						rule.getLeftHandRule().getOuterRuleMembrane()
								.getLabel())) {
					this.membrane = m;
					end = true;
				}
			}

			// The left membrane of this reaction is the skin
			if (this.membrane.isSkinMembrane())
				parentMembrane = null;
			else if (this.membrane instanceof CellLikeNoSkinMembrane)
				parentMembrane = ((CellLikeNoSkinMembrane) this.membrane)
						.getParentMembrane();

			// Build the list of reactants
			reactants = new HashMap<String, Long>();
			// Inside of left membrane [a]
			MultiSet<String> ruleContent = rule.getLeftHandRule()
					.getOuterRuleMembrane().getMultiSet();
			Iterator<String> itr = ruleContent.iterator();

			while (itr.hasNext()) {
				String r = itr.next();
				reactants.put(r, ruleContent.count(r));
			}

			// Outside of left membrane a[]
			outerReactants = new HashMap<String, Long>();

			ruleContent = rule.getLeftHandRule().getMultiSet();
			itr = ruleContent.iterator();
			while (itr.hasNext()) {
				String r = itr.next();
				outerReactants.put(r, ruleContent.count(r));
			}

			// Build the list of products
			// We don't care about the multiplicity
			products = new HashSet<String>();
			itr = rule.getRightHandRule().getOuterRuleMembrane().getMultiSet()
					.iterator();
			while (itr.hasNext()) {
				String r = itr.next();
				products.add(r);
			}
			outerProducts = new HashSet<String>();
			itr = rule.getRightHandRule().getMultiSet().iterator();
			while (itr.hasNext()) {
				String r = itr.next();
				outerProducts.add(r);
			}

			// And last but not least, calculate a, at time 0
			updateA(0);
		}
		/**
		 * @return the index
		 */
		public int getIndex() {
			return index;
		}

		/**
		 * @param index
		 *            the index to set
		 */
		public void setIndex(int index) {
			this.index = index;
		}

		/**
		 * @return The set of species that affect the value of a for this
		 *         reaction
		 */
		public Set<Pair<String, String>> dependsOn() {
			Set<Pair<String, String>> res = new HashSet<Pair<String, String>>();
			addInvolvedMultiSets(reactants.keySet(), outerReactants.keySet(), res);
//			Iterator<String> it = reactants.keySet().iterator();
//			while (it.hasNext()) {
//				String x = it.next();
//				Pair<String, String> p = new Pair<String, String>(x, membrane
//						.getLabel());
//				res.add(p);
//			}
//
//			it = outerReactants.keySet().iterator();
//			while (it.hasNext()) {
//				String x = it.next();
//				Pair<String, String> p = new Pair<String, String>(x,
//						parentMembrane.getLabel());
//				res.add(p);
//			}
			return res;
		}
		
		/*This method is used to perform the common functionality for dependsOn and affects methods, so that there's no need to repeat the same code.
		 * The first time this method is called (with the same resultingSet parameter) resultingSet should be an empty, recently-created set*/
		private void addInvolvedMultiSets(Set<String> innerObjects, Set<String> outerObjects, Set<Pair<String, String>> resultingSet){
			addInvolvedObjects(innerObjects, resultingSet, membrane.getLabel());
			if(parentMembrane!=null)
				addInvolvedObjects(outerObjects, resultingSet, parentMembrane.getLabel());
		}
		
		
		private void addInvolvedObjects(Set<String> objectsToInvolve, Set<Pair<String, String>> resultingSet, String label){
			Iterator<String> it = objectsToInvolve.iterator();
			while(it.hasNext()){
				String x = it.next();
				Pair<String, String> p = new Pair<String, String>(x, label);
				resultingSet.add(p);
			}
		}
		
		

		/**
		 * @return The set of species that change quantity when this reaction is
		 *         executed
		 */
		public Set<Pair<String, String>> affects() {
			Set<Pair<String, String>> res = new HashSet<Pair<String, String>>();
			res.addAll(this.dependsOn());
			addInvolvedMultiSets(products, outerProducts, res);			
//			Iterator<String> it = products.iterator();
//			while (it.hasNext()) {
//				String x = it.next();
//				Pair<String, String> p = new Pair<String, String>(x, membrane
//						.getLabel());
//				res.add(p);
//			}
//
//			it = outerProducts.iterator();
//			while (it.hasNext()) {
//				String x = it.next();
//				Pair<String, String> p = new Pair<String, String>(x,
//						parentMembrane.getLabel());
//				res.add(p);
//			}
			return res;

		}

		/**
		 * @return a
		 */
		public double getA() {
			return a;
		}

		/**
		 * @return the tau
		 */
		public double getTau() {
			return tau;
		}

		/**
		 * @param tau
		 *            the tau to set
		 */
		public void setTau(double tau) {
			this.tau = tau;
		}

		public void execute() {
			if (rule.isExecutable(membrane)) {
				if (parentMembrane == null)
					rule.execute(membrane, currentConfig.getEnvironment(), 1);
				else
					rule.execute(membrane, parentMembrane.getMultiSet(), 1);
			}
			//currentConfig.getEnvironment();
		}

		/**
		 * Calculates the value of a, according to the current content of the
		 * membranes in the system
		 * 
		 * @param t
		 *            Current time
		 */
		public void updateA(double t) {

			boolean a_not_zero = false;

			if (a != 0) {
				a_before_zero = a; // the last non-zero value of a
				a_not_zero = true;
			}

			MultiSet<String> content = membrane.getMultiSet();
			a = 1;

			Iterator<String> it = reactants.keySet().iterator();
			while (it.hasNext()) {
				String x = it.next();
				long n = reactants.get(x); // number of "x's" in the rule
				long X = 0; // multiplicity of x in the system
				if (!GeneString.isGeneString(x))
					// if x is a symbol, its multiplicity is the number of x
					// in the membrane
					X = content.count(x); 
				else {
					// if x is a string, its multiplicity is the number of 
					// substrings in the membrane that x matches
					GeneString gx = new GeneString(GeneString.getString(x));
					X = gx.count(content);
				}
					
				// if X == 0, a <- 0 and the reaction can not occur
				for (int i = 0; i < n; i++) {
					a *= (double) (X - i) / (i + 1);
				}

			}

			if (parentMembrane == null && !outerReactants.isEmpty())
				a = 0;
			else if (parentMembrane != null) {
				content = parentMembrane.getMultiSet();
				it = outerReactants.keySet().iterator();
				while (it.hasNext()) {
					String x = it.next();
					long n = outerReactants.get(x);
					long X = content.count(x);
					for (int i = 0; i < n; i++) {
						a *= (double) (X - i) / (i + 1);
					}
				}
			}

			a *= c;

			// if a is 0 but a wasn't zero before...
			if (a == 0 && a_not_zero)
				setT_1(t);
		}

		/**
		 * Generates a putative time tau for this reaction, according to an
		 * exponential distribution with parameter a
		 */
		public void generatePutativeTime() {
			// RandomElement e = new RanMT();
			// double r1 = e.raw();
			if (a == 0)
				tau = Double.MAX_VALUE;
			else {
				double r1 = Math.random();
				tau = (1 / a) * Math.log(1 / r1);
			}
		}

		@Override
		public String toString() {
			String res = "";
			Iterator<String> it = outerReactants.keySet().iterator();
			while (it.hasNext()) {
				res += it.next();
				if (it.hasNext())
					res += "+";
			}

			res += "[";
			it = reactants.keySet().iterator();
			while (it.hasNext()) {
				res += it.next();
				if (it.hasNext())
					res += " + ";
			}

			res += "] --> ["
					+ rule.getRightHandRule().getOuterRuleMembrane()
							.getMultiSet() + "]";
			return res + "membrane " + membrane + " parent membrane "
					+ parentMembrane;
			// return "("+index+","+tau+")";
		}

		@Override
		public int compareTo(Reaction o) {
			double res = this.tau - o.getTau();
			if (res < 0)
				return -1;
			else if (res > 0)
				return 1;

			return 0;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + index;
			return result;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			Reaction other = (Reaction) obj;
			if (index != other.index)
				return false;
			return true;
		}

		/**
		 * @param t_1
		 *            the t_1 to set
		 */
		public void setT_1(double t_1) {
			this.t_1 = t_1;
		}

		/**
		 * @return the t_1
		 */
		public double getT_1() {
			return t_1;
		}

		/**
		 * @return the a_before_zero
		 */
		public double getA_before_zero() {
			return a_before_zero;
		}

		/**
		 * @return The label of the left membrane of this reaction
		 */
		public String getLabel() {
			return membrane.getLabel();
		}

		/**
		 * @return the membrane
		 */
		public ChangeableMembrane getMembrane() {
			return membrane;
		}

		/**
		 * @return the globalIndex
		 */
		public int getGlobalIndex() {
			return globalIndex;
		}

		/**
		 * @param globalIndex
		 *            the globalIndex to set
		 */
		public void setGlobalIndex(int globalIndex) {
			this.globalIndex = globalIndex;
		}

	}

	/**
	 * @see org.gcn.plinguacore.simulator.ISimulator#isFinished()
	 *  In the case of this stochastic simulator, if the simulation is over (T > timeOut)
	 */
	@Override
	  public boolean isFinished() {
	      return T > timeOut;
	  }
	
	
	

}
