/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.gcn.plinguacore.simulator.cellLike.stochastic;

import java.util.Iterator;

import org.gcn.plinguacore.simulator.cellLike.stochastic.StochasticSimulator;
import org.gcn.plinguacore.util.psystem.Psystem;
import org.gcn.plinguacore.util.psystem.membrane.ChangeableMembrane;
import org.gcn.plinguacore.util.psystem.membrane.MembraneStructure;
import org.gcn.plinguacore.util.psystem.rule.IRule;


/**
 *  @author Research Group on Natural Computing (http://www.gcn.us.es)
 *
 *
 * 
 *         Simulator of stochastic P-systems based on Multicompartimental Direct
 *         Gillespie algorithm. In this simulator the reactions are stored in a
 *         list all together. The next reaction to be executed, as well as the
 *         next time to which the system evolves, are obtained with two random
 *         numbers and an exponential distribution whose parameter is a_0. All
 *         reactions are recalculated and updated every step, even if they are
 *         not affected by the last execution.
 */
public final class DirectStochasticSimulator extends StochasticSimulator {

	private static final long serialVersionUID = 3270787660596924883L;
	private double nextTime;

	/**
	 * Constructor. Just call the constructor of superclass...
	 * 
	 * @param psystem
	 *            The p-system to be simulated
	 */
	public DirectStochasticSimulator(Psystem psystem) {
		super(psystem);
	}

	/**
	 * @return The next reaction to be executed
	 * 
	 */
	@Override
	public StochasticSimulator.Reaction getNextReaction() {
		StochasticSimulator.Reaction nextReaction = null;
		Iterator<StochasticSimulator.Reaction> it = reactions.iterator();
		// a_0 = sum a_r, for every reaction r
		double a_0 = 0;
		while (it.hasNext()) {
			Reaction r = it.next();
			a_0 += r.getA();
		}
		double r1 = Math.random();
		double r2 = Math.random();

		// Time at which the next reaction is scheduled
		double tau = (1 / a_0) * Math.log(1 / r1);

		// The system time draw on tau
		nextTime = t + tau;

		// Selected reaction
		int mu = -1;
		double aux = 0;
		while (aux < r2 * a_0) {
			mu++;
			aux += reactions.get(mu).getA();
		}

		if (mu >= 0)
			nextReaction = reactions.get(mu);

		return nextReaction;

	}

	/**
	 * @return The next instant to which the system evolves
	 */
	@Override
	public double getNextTime() {
		return nextTime;
	}

	/**
	 * Updates the rest of the reactions, after executing nextReaction. In this
	 * classic algorithm, all reactions are updated
	 */
	@Override
	public void updateSystem() {
		// Update a for every reaction in the system
		Iterator<Reaction> it = reactions.iterator();
		while (it.hasNext()) {
			Reaction r = it.next();
			r.updateA(t);
		}
	}

	

}
