/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.gcn.plinguacore.simulator.cellLike.stochastic;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import org.gcn.plinguacore.util.IndexedPriorityQueue;
import org.gcn.plinguacore.util.Pair;
import org.gcn.plinguacore.util.psystem.Psystem;

/**
 *
 *  @author Research Group on Natural Computing (http://www.gcn.us.es)
 *
 *
 * 
 *         Simulator of stochastic P-systems based on the algorithm
 *         Multicompartimental Next Reaction Method. In this simulator the
 *         reactions are stored in a priority queue and a graph of dependencies
 *         is maintained. In this way, the next reaction is chosen in constant
 *         time, thanks to the priority queue, and only the affected reactions
 *         are updated in every step, thanks to the dependency graph.
 * 
 */
public class EfficientStochasticSimulator extends StochasticSimulator {

	private static final long serialVersionUID = 3270787660596924883L;

	private double nextTime; // time to which the system evolves, after
	// returning the next reaction
	private Reaction nextReaction; // next reaction to be executed

	private List<Reaction>[] G; // dependency graph
	private Map<String, IndexedPriorityQueue<Reaction>> P; // priority queues,
	// one for each
	// membrane

	private long seed = 1204198409;

	/**
	 * Constructor. Basically, all the data structures needed are created and
	 * initialized here.
	 * 
	 * @param psystem
	 *            The P-system to be simulated
	 */
	public EfficientStochasticSimulator(Psystem psystem) {
		super(psystem);

		// build indexed priority queues
		buildIndexedPriorityQueues();

		// generate dependency graph
		generateDependencyGraph();

	}

	/**
	 * Re-build the data structures used in this algorithm
	 */
	@Override
	public void reset() {
		super.reset();
		// build indexed priority queues
		buildIndexedPriorityQueues();
		// generate dependency graph
		generateDependencyGraph();
	}

	/**
	 * @return The next reaction to be executed in the system This class
	 *         implements the "Next Reaction Method". The reaction returned is
	 *         the minimum element stored in the indexed priority queues, that
	 *         is, the reaction whose putative time stored in P, is least
	 */
	@Override
	public Reaction getNextReaction() {
		nextReaction = null;
		Iterator<String> itm = P.keySet().iterator();
		while (itm.hasNext()) {
			String m = itm.next();
			IndexedPriorityQueue<Reaction> pm = P.get(m);
			Reaction min = pm.findMinimum();
			if (nextReaction == null || nextReaction.compareTo(min) > 0)
				nextReaction = min;
		}

		// the current time of the system is the putative time of the next
		// reaction
		if (nextReaction != null)
			nextTime = nextReaction.getTau();
		return nextReaction;
	}

	/**
	 * @return The next instant to which the system evolves
	 */
	@Override
	public double getNextTime() {
		return nextTime;
	}

	/**
	 * Recalculates a for the reactions needed, according to the dependency
	 * graph and the last reaction executed (next reaction) Also, updates the
	 * tau (scheduled time of execution) for those reactions.
	 */
	@Override
	public void updateSystem() {
		// For each edge (mu,alpha) in G
		List<Reaction> edges = G[nextReaction.getGlobalIndex()];
		if (edges != null && !edges.isEmpty()) {
			Iterator<Reaction> it = edges.iterator();
			while (it.hasNext()) {
				Reaction r_alpha = it.next();
				double a_alpha_old = r_alpha.getA();
				// (a) update a_alpha
				r_alpha.updateA(nextTime);

				// r_alpha.generatePutativeTime();
				// r_alpha.setTau(r_alpha.getTau()+nextTime);

				// (b) if mu != alpha
				// set tau_alpha <- (a_alpha,old/a_alpha,new)(tau_alpha-t) + t

				if (!nextReaction.equals(r_alpha)) {
					double tau_alpha;
					// the new a for r_alpha is 0
					// tau_alpha = infinity
					if (r_alpha.getA() == 0)
						tau_alpha = Double.MAX_VALUE;
					// the new a for r_alpha is not 0 but the previous one was
					else if (a_alpha_old == 0) {
						// previous t_alpha = infinity, so we need a new
						// putative time
						r_alpha.generatePutativeTime();
						// this is the first time that a_alpha is not zero
						if (r_alpha.getT_1() == 0)
							tau_alpha = nextTime + r_alpha.getTau();

						// this is not the first time
						else
							tau_alpha = (r_alpha.getA_before_zero() / r_alpha
									.getA())
									* (r_alpha.getTau() - r_alpha.getT_1())
									+ nextTime;

					}
					// neither a_alpha is zero nor a_alpha_old
					else
						tau_alpha = (a_alpha_old / r_alpha.getA())
								* (r_alpha.getTau() - nextTime) + nextTime;

					r_alpha.setTau(tau_alpha);
				}
				// (c) if mu = alpha
				// generate a random number ro, according to an exponential
				// distribution with parameter a_mu and set t_alpha = ro + t
				else {
					// RandomElement e = new Ranmar();
					// double r1 = e.raw();
					seed += 93801923 % 10298310;
					Random r = new Random(seed);
					double r1 = r.nextDouble();
					double ro = (1 / nextReaction.getA()) * Math.log(1 / r1);
					nextReaction.setTau(ro + nextTime);
				}
				// (d) replace the old t_alpha value in P with the new value
				IndexedPriorityQueue<Reaction> pm = P.get(r_alpha.getLabel());
				pm.update(pm.getNode(r_alpha.getIndex()), r_alpha);
				P.put(r_alpha.getLabel(), pm);
			}
		}

	}

	/*
	 * Creates and initializes the attribute P, a map that contains one indexed
	 * priority queue for each membrane in the psystem
	 */
	private void buildIndexedPriorityQueues() {
		// We need a priority queue for each membrane in the system
		P = new HashMap<String, IndexedPriorityQueue<Reaction>>();
		Map<String, List<Reaction>> aux = new HashMap<String, List<Reaction>>();

		// generate a putative time for each reaction
		Iterator<Reaction> itr = reactions.iterator();
		while (itr.hasNext()) {
			Reaction r = itr.next();
			r.generatePutativeTime();
		}

		// store the reactions in a priority queue
		Collections.sort(reactions);
		int globalIndex = 0;

		// we have to classify them and assign them an index first...
		itr = reactions.iterator();
		while (itr.hasNext()) {
			Reaction r = itr.next();
			r.setGlobalIndex(globalIndex++);
			String m = r.getLabel();
			List<Reaction> ml = null;
			if (!aux.containsKey(m))
				ml = new LinkedList<Reaction>();
			else
				ml = aux.get(m);

			r.setIndex(ml.size());
			ml.add(r);
			aux.put(m, ml);
		}

		Iterator<String> itm = aux.keySet().iterator();
		while (itm.hasNext()) {
			String m = itm.next();
			IndexedPriorityQueue<Reaction> pm = new IndexedPriorityQueue<Reaction>(
					aux.get(m));
			P.put(m, pm);
		}

	}

	/*
	 * Generates a directed graph G=(V,E), where V = {reactions} and E =
	 * {(r_i,r_j): r_i.affects() intersection r_j.dependsOn() != empty} We
	 * represent this graph by using adjacency lists. Since the number of
	 * reactions is fixed for a system, we can use an array of lists.
	 */
	private void generateDependencyGraph() {
		G = new List[reactions.size()];

		Iterator<Reaction> it = reactions.iterator();
		while (it.hasNext()) {
			Reaction r_i = it.next();
			List<Reaction> edges = new LinkedList<Reaction>();
			Iterator<Reaction> it2 = reactions.iterator();
			while (it2.hasNext()) {
				Reaction r_j = it2.next();
				Set<Pair<String, String>> intersection = r_i.affects();
				intersection.retainAll(r_j.dependsOn());
				if (!intersection.isEmpty())
					edges.add(r_j);
			}
			// If for some strange reason, the self edges from r_i to r_i
			// are not included, include them as well. That happens in the
			// case that reactants(r_i) = empty
			if (!edges.contains(r_i))
				edges.add(r_i);
			G[r_i.getGlobalIndex()] = edges;
		}
	}
}
