/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.gcn.plinguacore.simulator.kernel;



import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import org.gcn.plinguacore.simulator.AbstractSelectionExecutionSimulator;

import org.gcn.plinguacore.util.MultiSet;
import org.gcn.plinguacore.util.Pair;
import org.gcn.plinguacore.util.psystem.Configuration;
import org.gcn.plinguacore.util.psystem.Psystem;
import org.gcn.plinguacore.util.psystem.kernel.membrane.KernelLikeMembraneStructure;
import org.gcn.plinguacore.util.psystem.membrane.ChangeableMembrane;
import org.gcn.plinguacore.util.psystem.membrane.Membrane;
import org.gcn.plinguacore.util.psystem.membrane.MembraneStructure;
import org.gcn.plinguacore.util.psystem.rule.IKernelRule;
import org.gcn.plinguacore.util.psystem.rule.IRule;
import org.gcn.plinguacore.util.psystem.rule.RulesSet;
import org.gcn.plinguacore.util.psystem.rule.kernel.DivisionKernelLikeRule;
import org.gcn.plinguacore.util.psystem.rule.kernel.EvolutionCommunicationKernelLikeRule;
import org.gcn.plinguacore.util.psystem.tissueLike.membrane.TissueLikeMembrane;
import org.gcn.plinguacore.util.psystem.tissueLike.membrane.TissueLikeMembraneStructure;



import java.util.List;

public class KernelSimulator extends AbstractSelectionExecutionSimulator {

	/**
	 * 
	 */
	private static final int DIVISION_STAGE=1;
	private static final int EVOLUTION_AND_COMMUNICATION_STAGE=0;
	private static final long serialVersionUID = -1556761252509150318L;
	private static final int STATE_SELECT_DIVISION=10;
	private static final int STATE_SELECT_COMMUNICATION=20;
	private int state;
	protected KernelLikeMembraneStructure structure;
	private Map<Integer,Pair<IRule,ChangeableMembrane>>membranesToDivide;
	private List<Pair<EvolutionCommunicationKernelLikeRule, ChangeableMembrane>> evolutionRules;
	
	

	public KernelSimulator(Psystem psystem) {
		super(psystem);
		evolutionRules = new LinkedList<Pair<EvolutionCommunicationKernelLikeRule, ChangeableMembrane>>();
		membranesToDivide=new HashMap<Integer,Pair<IRule,ChangeableMembrane>>();
	
	}

	@Override
	protected String getHead(ChangeableMembrane m) {
		// TODO Auto-generated method stub
		String str;
		str="CELL ID: "+m.getId();
		str += ", Label: " + m.getLabelObj();
		return str;
	}
	
	

	@Override
	protected void microStepInit() {
		// TODO Auto-generated method stub
		super.microStepInit();
		structure = (KernelLikeMembraneStructure)getPsystem().getMembraneStructure();
		membranesToDivide.clear();
		evolutionRules.clear();
		state=STATE_SELECT_COMMUNICATION;
	}

	@Override
	protected void microStepSelectRules(Configuration cnf, Configuration tmpCnf) {
		// TODO Auto-generated method stub
		for (int i=EVOLUTION_AND_COMMUNICATION_STAGE;i<=DIVISION_STAGE;i++)
		{
			if (i==DIVISION_STAGE)
				state = STATE_SELECT_DIVISION;
			Iterator<? extends Membrane> it = tmpCnf.getMembraneStructure().getAllMembranes().iterator();
			Iterator<? extends Membrane> it1 = cnf.getMembraneStructure().getAllMembranes().iterator();
			while(it.hasNext())
			{
				ChangeableMembrane tempMembrane = (ChangeableMembrane) it.next();
				ChangeableMembrane m = (ChangeableMembrane)it1.next();
				microStepSelectRules(m,tempMembrane);
			}
		}
	}

	@Override
	protected void microStepSelectRules(ChangeableMembrane m,
			ChangeableMembrane temp) {
		Iterator<IRule> it = getPsystem().getRules().iterator(temp.getLabel(),
				temp.getCharge(),true);

		while (it.hasNext()) {
			IRule r = it.next();
			
			structure = (KernelLikeMembraneStructure)((TissueLikeMembrane)m).getStructure();
			
			DivisionKernelLikeRule r1 = (DivisionKernelLikeRule)r;
			r1.setMembraneStructure(structure);
			if (state==STATE_SELECT_COMMUNICATION && (r instanceof EvolutionCommunicationKernelLikeRule) && !isDivision(r))
			{					
				selectEvolutionCommunicationRule(m, temp, r1);
			}
			else
			if (state==STATE_SELECT_DIVISION && isDivision(r) /*&& !membranesToCommunicate.contains(m.getId())*/ && !membranesToDivide.containsKey(m.getId()))
			{
				selectDivisionRule(m, temp, r);
			}
		}
		
	}

	protected void selectDivisionRule(ChangeableMembrane m,
			ChangeableMembrane temp, IRule r) {
		long count = r.countExecutions(temp);
		if (count>0)
		{
			count=1;
			selectRule(r, m, count);
			membranesToDivide.put(m.getId(),new Pair<IRule,ChangeableMembrane>(r,m));
			removeLeftHandRuleObjects(temp, r, count);
			
		}
	}

	protected void selectEvolutionCommunicationRule(ChangeableMembrane m,
			ChangeableMembrane temp, DivisionKernelLikeRule r1) {
		r1.setMembraneStructure(structure);
		long count = r1.countExecutions(temp);
		if (count>0)
		{

				selectRule(r1, m, count);
				removeLeftHandRuleObjects(temp, r1,count);
			evolutionRules.add(new Pair<EvolutionCommunicationKernelLikeRule, ChangeableMembrane>((EvolutionCommunicationKernelLikeRule)r1, m));
		}
	}

	private boolean isDivision(IRule r) {
		
		// TODO Auto-generated method stub
		return !(r instanceof EvolutionCommunicationKernelLikeRule);
	}

	protected TissueLikeMembrane getMembraneByLabel(
			DivisionKernelLikeRule r1,
			TissueLikeMembraneStructure structure) {
		Iterator<TissueLikeMembrane> it = structure.iterator(r1.getRightHandRule().getOuterRuleMembrane().getLabelObj().getLabelID());
		TissueLikeMembrane tlm = null;
		if (it.hasNext()){
			tlm = it.next();
		}
		
		return tlm;
	}

	protected boolean isEvolution(IRule r){
		return r.getLeftHandRule().getOuterRuleMembrane().getLabel().equals(r.getRightHandRule().getOuterRuleMembrane().getLabel());
	}
	
	protected RulesSet filterRules(Iterator<IRule> iterator, ChangeableMembrane m) {
		// TODO Auto-generated method stub
		RulesSet returnedRuleSet = new RulesSet();
		while(iterator.hasNext()){
			IKernelRule rule= (IKernelRule) iterator.next();
			if(rule.guardEvaluates(m))
				returnedRuleSet.add(rule);
		}
		return returnedRuleSet;
	}

	@Override
	protected void microStepExecuteRules() {
		executeEvolutionCommunicationRules();
		executeDivisionRules();		
	}

	protected void executeDivisionRules() {
		Iterator<Pair<IRule,ChangeableMembrane>>it1 = membranesToDivide.values().iterator();
		while(it1.hasNext())
		{
			Pair<IRule,ChangeableMembrane>p = it1.next();
			p.getFirst().execute(p.getSecond(), null);
		}
	}

	protected void executeEvolutionCommunicationRules() {
		for(Pair<EvolutionCommunicationKernelLikeRule, ChangeableMembrane> pair: evolutionRules){
			EvolutionCommunicationKernelLikeRule r1 = pair.getFirst();
			ChangeableMembrane m = pair.getSecond();
			long count = r1.countExecutions(m);
			TissueLikeMembrane tlm = getMembraneByLabel(r1, structure);
			if (tlm != null)
				r1.setRightHandMembrane(tlm);
			r1.execute(structure.getCell(m.getId()), currentConfig.getEnvironment(), count);
		}
	}

	@Override
	protected void printInfoMembraneShort(MembraneStructure membraneStructure) {
		// TODO Auto-generated method stub
		Iterator<? extends Membrane>it = membraneStructure.getAllMembranes().iterator();
		while(it.hasNext())
			printInfoMembrane((ChangeableMembrane)it.next());
	}

	@Override
	protected void printInfoMembrane(ChangeableMembrane membrane) {
		// TODO Auto-generated method stub
		TissueLikeMembrane tlm = (TissueLikeMembrane)membrane;
		if (!tlm.getLabel().equals(tlm.getStructure().getEnvironmentLabel()))
		{
			getInfoChannel().println("    " + getHead(membrane));
			getInfoChannel().println("    Multiset: " + membrane.getMultiSet());
			getInfoChannel().println();
		}
	}
	

	@Override
	protected void removeLeftHandRuleObjects(ChangeableMembrane membrane,
			IRule r, long count) {
		// TODO Auto-generated method stub
		MultiSet<String>ms = r.getLeftHandRule().getOuterRuleMembrane().getMultiSet();
		if (!ms.isEmpty())
			membrane.getMultiSet().subtraction(ms, count);
	}

	

	

}
