/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.gcn.plinguacore.simulator.cellLike.stochastic;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcn.plinguacore.simulator.cellLike.stochastic.StochasticSimulator;
import org.gcn.plinguacore.util.IndexedPriorityQueue;
import org.gcn.plinguacore.util.Pair;
import org.gcn.plinguacore.util.psystem.Psystem;


public class DeterministicWaitingTimesSimulator extends StochasticSimulator {

	
	private static final long serialVersionUID = 5072278426632772554L;
	private double nextTime; // time to which the system evolves, after
	// returning the next reaction
	private Reaction nextReaction; // next reaction to be executed
	private List<Reaction>[] G; // dependency graph
	private Map<String, IndexedPriorityQueue<Reaction>> P; // priority queues,
	// one for each
	// membrane
	
	
	public DeterministicWaitingTimesSimulator(Psystem psystem) {
		super(psystem);
		// build indexed priority queues
		buildIndexedPriorityQueues();

		// generate dependency graph
		generateDependencyGraph();	}

	@Override
	public Reaction getNextReaction() {
		nextReaction = null;
		Iterator<String> itm = P.keySet().iterator();
		while (itm.hasNext()) {
			String m = itm.next();
			IndexedPriorityQueue<Reaction> pm = P.get(m);
			Reaction min = pm.findMinimum();
			if (nextReaction == null || nextReaction.compareTo(min) > 0)
				nextReaction = min;
		}

		// the next time of the system is current time + waiting time
		// of the next reaction
		if (nextReaction != null)
			nextTime = nextTime + nextReaction.getTau();
		return nextReaction;		
	}

	@Override
	public double getNextTime() {
		return nextTime;

	}

	@Override
	public void updateSystem() {
		// First, update waiting time for all reactions, by subtracting the
		// tau of nextReaction
		Iterator<Reaction> it = reactions.iterator();
		while (it.hasNext()) {
			Reaction r = it.next();
			r.setTau(r.getTau()-nextReaction.getTau());
		}
		
		// For each reaction affected by execution of nextReaction 
		// recalculate a and tau
		List<Reaction> edges = G[nextReaction.getGlobalIndex()];
		if (edges != null && !edges.isEmpty()) {
			it = edges.iterator();
			while (it.hasNext()) {
				Reaction r = it.next();
				r.updateA(nextTime);
				double a = r.getA();
				if (a==0)
					r.setTau(Double.MAX_VALUE);
				else
					r.setTau(1/a);
				// reorganize the priority queue
				IndexedPriorityQueue<Reaction> pm = P.get(r.getLabel());
				pm.update(pm.getNode(r.getIndex()), r);
				P.put(r.getLabel(), pm);
			}
		}
	}
	
	/**
	 * Re-build the data structures used in this algorithm
	 */
	@Override
	public void reset() {
		super.reset();
		// build indexed priority queues
		buildIndexedPriorityQueues();
		// generate dependency graph
		generateDependencyGraph();
	}

	/*
	 * Creates and initializes the attribute P, a map that contains one indexed
	 * priority queue for each membrane in the psystem
	 */
	private void buildIndexedPriorityQueues() {
		// We need a priority queue for each membrane in the system
		P = new HashMap<String, IndexedPriorityQueue<Reaction>>();
		Map<String, List<Reaction>> aux = new HashMap<String, List<Reaction>>();

		// compute the waiting time associated with rule r as 1/v(r)
		// v(r) is the velocity, that is, a
		Iterator<Reaction> itr = reactions.iterator();
		while (itr.hasNext()) {
			Reaction r = itr.next();
			double a = r.getA();
			if (a == 0)
				r.setTau(Double.MAX_VALUE);
			else
				r.setTau(1/a);
		}

		// store the reactions in a priority queue
		Collections.sort(reactions);
		int globalIndex = 0;

		// we have to classify them and assign them an index first...
		itr = reactions.iterator();
		while (itr.hasNext()) {
			Reaction r = itr.next();
			r.setGlobalIndex(globalIndex++);
			String m = r.getLabel();
			List<Reaction> ml = null;
			if (!aux.containsKey(m))
				ml = new LinkedList<Reaction>();
			else
				ml = aux.get(m);

			r.setIndex(ml.size());
			ml.add(r);
			aux.put(m, ml);
		}

		Iterator<String> itm = aux.keySet().iterator();
		while (itm.hasNext()) {
			String m = itm.next();
			IndexedPriorityQueue<Reaction> pm = new IndexedPriorityQueue<Reaction>(
					aux.get(m));
			P.put(m, pm);
		}

	}

	/*
	 * Generates a directed graph G=(V,E), where V = {reactions} and E =
	 * {(r_i,r_j): r_i.affects() intersection r_j.dependsOn() != empty} We
	 * represent this graph by using adjacency lists. Since the number of
	 * reactions is fixed for a system, we can use an array of lists.
	 */
	private void generateDependencyGraph() {
		G = new List[reactions.size()];

		Iterator<Reaction> it = reactions.iterator();
		while (it.hasNext()) {
			Reaction r_i = it.next();
			List<Reaction> edges = new LinkedList<Reaction>();
			Iterator<Reaction> it2 = reactions.iterator();
			while (it2.hasNext()) {
				Reaction r_j = it2.next();
				Set<Pair<String, String>> intersection = r_i.affects();
				intersection.retainAll(r_j.dependsOn());
				if (!intersection.isEmpty())
					edges.add(r_j);
			}
			// If for some strange reason, the self edges from r_i to r_i
			// are not included, include them as well. That happens in the
			// case that reactants(r_i) = empty
			if (!edges.contains(r_i))
				edges.add(r_i);
			G[r_i.getGlobalIndex()] = edges;
		}
	}
}
