/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.gcn.plinguacore.parser.output.binary;


import java.io.IOException;

import java.util.ArrayList;
import java.util.Iterator;

import java.util.List;



import org.gcn.plinguacore.util.psystem.rule.IRule;
import org.gcn.plinguacore.util.psystem.rule.checkRule.CheckRule;
import org.gcn.plinguacore.util.psystem.rule.checkRule.specificCheckRule.NoDissolution;
import org.gcn.plinguacore.util.psystem.rule.checkRule.specificCheckRule.NoDivision;
import org.gcn.plinguacore.util.psystem.rule.checkRule.specificCheckRule.NoEvolution;
import org.gcn.plinguacore.util.psystem.rule.checkRule.specificCheckRule.NoLeftExternalMultiSet;

/**
 * This class writes an active membranes P system in a binary file
 * 
 *  @author Research Group on Natural Computing (http://www.gcn.us.es)
 * 
 */
class AMBinaryOutputParser extends AbstractBinaryOutputParser {


	

	

	private List<IRule> evolutionRules = null;
	private List<IRule> sendInRules = null;
	private List<IRule> sendOutRules = null;
	private List<IRule> divisionRules = null;
	private List<IRule> dissolutionRules = null;

	private static final CheckRule noEvolutionRule = new NoEvolution(
			);

	private static final CheckRule noDivisionRule = new NoDivision(
			);

	private static final CheckRule noDissolutionRule = new NoDissolution(
			);

	private static final CheckRule noLeftExternalMultiSet = new NoLeftExternalMultiSet(
			);

	public AMBinaryOutputParser() {
	
		
		
		evolutionRules = new ArrayList<IRule>();
		sendInRules = new ArrayList<IRule>();
		sendOutRules = new ArrayList<IRule>();
		divisionRules = new ArrayList<IRule>();
		dissolutionRules = new ArrayList<IRule>();
	}

	
	

	private void readRules() {
		evolutionRules.clear();
		sendInRules.clear();
		sendOutRules.clear();
		divisionRules.clear();
		dissolutionRules.clear();

		Iterator<IRule> it = getPsystem().getRules().iterator();

		while (it.hasNext()) {
			IRule r = it.next();

			if (!noEvolutionRule.checkRule(r))
				evolutionRules.add(r);
			else if (!noDivisionRule.checkRule(r))
				divisionRules.add(r);
			else if (!noDissolutionRule.checkRule(r))
				dissolutionRules.add(r);
			else if (noLeftExternalMultiSet.checkRule(r))
				sendOutRules.add(r);
			else
				sendInRules.add(r);

		}
	}
	
	
	

	private void writeSendInRules() throws IOException {
		getStream().writeChar(sendInRules.size());
		Iterator<IRule> it = sendInRules.iterator();
		while(it.hasNext())
		{
			IRule r = it.next();
			writeRuleHead(r);
			writeCharge(r.getRightHandRule().getOuterRuleMembrane().getCharge());
			writeObjects(r.getLeftHandRule().getMultiSet(),false,true);
			writeObjects(r.getRightHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
		}
	}
	
	private void writeDissolutionRules() throws IOException {
		getStream().writeChar(dissolutionRules.size());
		Iterator<IRule> it = dissolutionRules.iterator();
		while(it.hasNext())
		{
			IRule r = it.next();
			writeRuleHead(r);
			writeObjects(r.getLeftHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
			if (!r.getRightHandRule().getMultiSet().isEmpty())
				writeObjects(r.getRightHandRule().getMultiSet(),false,true);
			else
				writeObjects(r.getRightHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
		}
	}
	
	private void writeDivisionRules() throws IOException {
		getStream().writeChar(divisionRules.size());
		Iterator<IRule> it = divisionRules.iterator();
		while(it.hasNext())
		{
			IRule r = it.next();
			writeRuleHead(r);
			writeCharge(r.getRightHandRule().getOuterRuleMembrane().getCharge());
			writeCharge(r.getRightHandRule().getSecondOuterRuleMembrane().getCharge());
			writeObjects(r.getLeftHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
			writeObjects(r.getRightHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
			writeObjects(r.getRightHandRule().getSecondOuterRuleMembrane().getMultiSet(),false,true);
		}
	}
	
	
	private void writeSendOutRules() throws IOException {
		getStream().writeChar(sendOutRules.size());
		Iterator<IRule> it = sendOutRules.iterator();
		while(it.hasNext())
		{
			IRule r = it.next();
			writeRuleHead(r);
			writeCharge(r.getRightHandRule().getOuterRuleMembrane().getCharge());
			writeObjects(r.getLeftHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
			writeObjects(r.getRightHandRule().getMultiSet(),false,true);
		}
	}
	
	private void writeEvolutionRules() throws IOException {
		getStream().writeChar(evolutionRules.size());
		Iterator<IRule> it = evolutionRules.iterator();
		while (it.hasNext()) {
			IRule r = it.next();
			writeRuleHead(r);
			writeObjects(r.getLeftHandRule().getOuterRuleMembrane().getMultiSet(),false,true);
			getStream().writeChar(r.getRightHandRule().getOuterRuleMembrane().getMultiSet().entrySet().size());
			writeObjects(r.getRightHandRule().getOuterRuleMembrane().getMultiSet(),true,false);
		}
	}
	@Override
	protected void writeRules() throws IOException {
		readRules();
		writeEvolutionRules();
		writeSendInRules();
		writeSendOutRules();
		writeDissolutionRules();
		writeDivisionRules();
	}




	@Override
	protected void writeHead() throws IOException {
		writeHead(ACTIVE_MEMBRANES_ID);

		
		
	}

	
	
	
	


}
