/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.gcn.plinguacore.parser.output.promela;


import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.util.*;

import org.gcn.plinguacore.parser.output.OutputParser;
import org.gcn.plinguacore.util.MultiSet;
import org.gcn.plinguacore.util.psystem.AlphabetObject;
import org.gcn.plinguacore.util.psystem.Psystem;
import org.gcn.plinguacore.util.psystem.membrane.Membrane;
import org.gcn.plinguacore.util.psystem.rule.IKernelRule;
import org.gcn.plinguacore.util.psystem.rule.IRule;
import org.gcn.plinguacore.util.psystem.rule.guard.IGuardVisitor;
import org.gcn.plinguacore.util.psystem.rule.kernel.KernelLikePsystem;
import org.stringtemplate.v4.ST;
import org.stringtemplate.v4.STGroup;
import org.stringtemplate.v4.STGroupFile;
/**
 * This class translates a P system into a promela code file
 * 
 *  @author Research Group on Natural Computing (http://www.gcn.us.es)
 * 
 */
public class PromelaOutputParser extends OutputParser {

	private KernelLikePsystem kPsystem;
	private AbstractMap<String, Integer> indicesBySymbol;
	private AbstractMap<String, Membrane> membranesByLabel;
	private AbstractMap<String, Integer> activeMembraneTypesByLabel;
	private AbstractMap<String, Integer> membraneTypesByLabel;
	private AbstractMap<String, List<IKernelRule>> rulesByMembraneLabel;
	private STGroup template;
	
	@Override
	public final boolean parse(Psystem psystem, OutputStream stream) {
		throw new UnsupportedOperationException();
	}

	@Override
	public final boolean parse(Psystem psystem, Writer stream) {
		if (!psystem.getAbstractPsystemFactory().getModelName().equals("kernel_psystems"))
			return false;
		else {
			
			kPsystem = (KernelLikePsystem) psystem;
			kPsystem.getAlphabet();
			indicesBySymbol = new HashMap<>();
			membranesByLabel = new HashMap<>();
			activeMembraneTypesByLabel = new HashMap<>();
			membraneTypesByLabel = new HashMap<>();
			rulesByMembraneLabel = new HashMap<>();
			template = new STGroupFile(getClass().getPackage().getName().replace('.', '/') + "/templates/sKPSystem.stg");
			
			
			
			int membraneType = 1;
			for(Membrane m : kPsystem.getMembraneStructure().getAllMembranes()) {
				List<IKernelRule> ruleList = new ArrayList<>();
				Iterator<IRule> ruleIterator = psystem.getRules().iterator(m.getLabel(), 0);
				while(ruleIterator.hasNext()) {
					ruleList.add((IKernelRule) ruleIterator.next());
				}
				
				membranesByLabel.put(m.getLabel(), m);
				rulesByMembraneLabel.put(m.getLabel(), ruleList);
				membraneTypesByLabel.put(m.getLabel(), membraneType);
				if(ruleList.size() > 0) {
					activeMembraneTypesByLabel.put(m.getLabel(), membraneType++);
				}
				else membraneType++;
			}
			
			Iterator<AlphabetObject> objectIterator = kPsystem.getAlphabet().iterator();
			for(int index = 0; objectIterator.hasNext(); index++) {
				indicesBySymbol.put(objectIterator.next().toString(), index);
			}
			
			//((TissueLikeMembrane)kPsystem.getMembraneStructure().getMembrane(1)).getMultiSet().;
			//rulesByMembraneLabel.get("1").get(2).getRightHandRule().getSecondOuterRuleMembrane()
			
			ST mainStructure = buildMainStructureTemplate();
			
			try {
				stream.write(mainStructure.render());
				stream.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
			
		return true;
	}
	
	protected ST buildMainStructureTemplate() {
		ST mainStructure = template.getInstanceOf("MainStructure");
		
		mainStructure.add("symbols", indicesBySymbol.keySet());
		mainStructure.add("symbolsMapping", buildSymbolsMapping());
		mainStructure.add("procRunner", buildProcRunners());
		mainStructure.add("membraneProcTypes", buildMembraneProcTypes());
		mainStructure.add("initialMultiSets", buildInitialMultiSets());
		
		return mainStructure;
	}
	
	protected ST buildSymbolsMapping() {
		 return PromelaSTHelper.buildSymbolsMapping(template, indicesBySymbol.keySet(), indicesBySymbol.values(), indicesBySymbol.size());
		
	}
	
	protected ST buildProcRunners() {
		ST procRunner = template.getInstanceOf("ProcRunner");
		procRunner.add("membraneTypes", activeMembraneTypesByLabel.values());
		procRunner.add("membraneLabels", activeMembraneTypesByLabel.keySet());
		return procRunner;
	}
	
	protected ST buildInitialMultiSets() {
		Map<String, MultiSet<String>> intialMultiSets = kPsystem.getInitialMultiSets();
		List<Integer> membraneTypes = new ArrayList<>();
		List<Collection<String>> symbolsList = new ArrayList<>();
		List<Collection<Long>> countsList = new ArrayList<>();
		
		for(String label : intialMultiSets.keySet()) {
			membraneTypes.add(membraneTypesByLabel.get(label));
		}
		
		for(String label : intialMultiSets.keySet()) {
			symbolsList.add(intialMultiSets.get(label).entrySet());
		}
		
		for(String label : intialMultiSets.keySet()) {
			MultiSet<String> multiSet = intialMultiSets.get(label);
			List<Long> counts = new ArrayList<>();
			for(String symbol : multiSet.entrySet()) {
				counts.add(multiSet.count(symbol));
			}
			countsList.add(counts);
		}
			
		return PromelaSTHelper.buildInitialMultiSets(template, getRange(0, intialMultiSets.size() - 1), membraneTypes, symbolsList, countsList, intialMultiSets.size());
	}
	
	protected List<String> buildMembraneProcTypes() {	
		List<String> membraneProcTypes = new ArrayList<String>();
		for(Membrane m : membranesByLabel.values()) {
			if(hasRules(m.getLabel())) {
				ST membraneProcType = template.getInstanceOf("MembraneProcType");
				membraneProcType.add("label", m.getLabel());
				membraneProcType.add("guardComputations",  buildGuardComputations(m.getLabel()).render());
				membraneProcType.add("rewritingAndCommRules", buildRewritingAndCommRules(m.getLabel()).render());
				//membraneProcType.add("divisionRules", "Division Rules");
				membraneProcTypes.add(membraneProcType.render());
			}
		}
		
		return membraneProcTypes;	
	}
	
	protected ST buildGuardComputations(String membraneLabel) {	
		ST guardComputationsTemplate = template.getInstanceOf("GuardComputations");
		List<IKernelRule> rules = rulesByMembraneLabel.get(membraneLabel);
		guardComputationsTemplate.add("ruleIndices", getRange(1, rules.size()));
		guardComputationsTemplate.add("guardEvals", buildGuardEvals(rules));
		guardComputationsTemplate.add("guardIfs", buildGuardIfs(rules));

		return guardComputationsTemplate;	
	}
	
	protected List<String> buildGuardEvals(List<IKernelRule> rules) {	
		List<String> guardEvals = new ArrayList<>();
		IGuardVisitor guardVisitor = new PromelaGuardVisitor(template);
		for(IKernelRule r : rules) {
			guardEvals.add(r.getGuard() != null ? r.getGuard().accept(guardVisitor) : "true");
		}
		
		return guardEvals;	
	}
	
	protected List<Boolean> buildGuardIfs(List<IKernelRule> rules) {	
		List<Boolean> guardIfs = new ArrayList<>();
		for(IKernelRule r : rules) {
			guardIfs.add(r.getGuard() != null ? true : false);
		}
		
		return guardIfs;	
	}
	
	protected ST buildRewritingAndCommRules(String membraneLabel) {	
		ST rulesTemplate = template.getInstanceOf("RewritingAndCommRules");
		
		List<IKernelRule> rules = rulesByMembraneLabel.get(membraneLabel);
		List<String> lefts = new ArrayList<>();
		List<String> rights = new ArrayList<>();
		
		for(IKernelRule rule : rules) {
			lefts.add(buildRuleLhs(rule));
			rights.add(buildRuleRhs(rule));
		}
		
		rulesTemplate.add("ruleIndices", getRange(1, rules.size()));
		rulesTemplate.add("ruleLhsList", lefts);
		rulesTemplate.add("ruleRhsList", rights);
		
		return rulesTemplate;	
	}
	
	protected String buildRuleLhs(IKernelRule rule) {
		// build left hand side symbols and multiplicity
		MultiSet<String> multiSet = rule.getLeftHandRule().getOuterRuleMembrane().getMultiSet();
		Set<String> symbols = multiSet.entrySet();
		List<Long> counts = new ArrayList<>();
		
		for(Object o : symbols) {
			counts.add(multiSet.count(o));
		}
	
		return PromelaSTHelper.buildRuleLhs(template, symbols, counts).render();
	}
	
	protected String buildRuleRhs(IKernelRule rule) {
		// build left hand side symbols and multiplicity
		String lhsMembraneLabel = rule.getLeftHandRule().getOuterRuleMembrane().getLabel();
		MultiSet<String> lhsMultiSet = rule.getLeftHandRule().getOuterRuleMembrane().getMultiSet();
		Set<String> lhsSymbols = lhsMultiSet.entrySet();
		List<Long> lhsCounts = new ArrayList<>();
		
		for(Object o : lhsSymbols) {
			lhsCounts.add(lhsMultiSet.count(o));
		}
		
		// build left hand side symbols, multiplicity and communication parameters
		String rhsMembraneLabel1 = rule.getRightHandRule().getOuterRuleMembrane().getLabel();
		String rhsMembraneLabel2 = rule.getRightHandRule().getSecondOuterRuleMembrane() != null ? rule.getRightHandRule().getSecondOuterRuleMembrane().getLabel() : "";
		MultiSet<String> rhsMultiSet1 = rule.getRightHandRule().getOuterRuleMembrane().getMultiSet();
		MultiSet<String> rhsMultiSet2 = rule.getRightHandRule().getSecondOuterRuleMembrane() != null ? rule.getRightHandRule().getSecondOuterRuleMembrane().getMultiSet() : null;
		List<String> rhsRewSymbols = new ArrayList<>();
		List<Long> rhsRewCounts = new ArrayList<>();
		List<Integer> rhsCommMembraneTypes = new ArrayList<>();
		List<String> rhsCommSymbols = new ArrayList<>();
		List<Long> rhsCommCounts = new ArrayList<>();
		
		if(rhsMultiSet1 != null) {
			if(lhsMembraneLabel.equals(rhsMembraneLabel1)) {
				rhsRewSymbols.addAll(rhsMultiSet1.entrySet());
			
				for(Object o : rhsMultiSet1.entrySet()) {
					rhsRewCounts.add(rhsMultiSet1.count(o));
				}
			}
			else {
				rhsCommMembraneTypes.add(membraneTypesByLabel.get(rhsMembraneLabel1));
				
				rhsCommSymbols.addAll(rhsMultiSet1.entrySet());
				
				for(Object o : rhsMultiSet1.entrySet()) {
					rhsCommCounts.add(rhsMultiSet1.count(o));
				}
			}
		}
		
		if(rhsMultiSet2 != null) {
			if(lhsMembraneLabel.equals(rhsMembraneLabel2)) {
				rhsRewSymbols.addAll(rhsMultiSet2.entrySet());
			
				for(Object o : rhsMultiSet2.entrySet()) {
					rhsRewCounts.add(rhsMultiSet2.count(o));
				}
			}
			else {
				rhsCommMembraneTypes.add(membraneTypesByLabel.get(rhsMembraneLabel2));
				
				rhsCommSymbols.addAll(rhsMultiSet2.entrySet());
				
				for(Object o : rhsMultiSet2.entrySet()) {
					rhsCommCounts.add(rhsMultiSet2.count(o));
				}
			}
		}
		
		return PromelaSTHelper.buildRuleRhs(template, lhsSymbols, lhsCounts, rhsRewSymbols, rhsRewCounts, rhsCommMembraneTypes, rhsCommSymbols, rhsCommCounts).render();
	}
	
	private boolean hasRules(String membraneLabel) {
		return rulesByMembraneLabel.get(membraneLabel).size() > 0;
	}
	
	private List<Integer> getRange(int min, int max) {
		List<Integer> values = new ArrayList<>();
		for(int i = min; i <= max; i++) {
			values.add(i);
		}
		
		return values;
	}
}
