/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.gcn.plinguacore.util.psystem.spiking.membrane;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.gcn.plinguacore.util.InfiniteMultiSet;
import org.gcn.plinguacore.util.psystem.cellLike.membrane.CellLikeSkinMembrane;
import org.gcn.plinguacore.util.psystem.membrane.ChangeableMembrane;
import org.gcn.plinguacore.util.psystem.membrane.Membrane;
import org.gcn.plinguacore.util.psystem.membrane.MembraneStructure;

public class SpikingMembraneStructure implements MembraneStructure {

	
	private Map<Integer,SpikingMembrane>cellsById;
	private Map<String,List<SpikingMembrane>>cellsByLabel;
	private SpikingEnvironment environment = null;
	private Integer input = null;
	private Map<String,Set<String>> dictionary = null;
	private Map<Integer,Integer> cloneMap = null;
	private Map<Integer,Set<Integer>> graph = null;
	private Map<Integer,Set<Integer>> rgraph = null;

	// the output membranes are in two ways calculated:
	// - as the predecessors to the environment (for spiking reasons)
	// - as members of the following set (for effciency reasons)
	
	private Set<Integer> output = null;
	
	// the next attributes are needed only in terms of simulation parameters
	
	private boolean showBinarySequence = false;
	private ArrayList showNaturalSequence = null;
	private boolean showSummatories = false;
	private int sequentialMode = 0;
	private int asynchMode = 0;
	private Map<String, Long> asynchValidConfiguration = null; 
		
	public SpikingMembraneStructure(String envLabel)
	{
		super();
		

		cellsById = new LinkedHashMap<Integer,SpikingMembrane>();
		cellsByLabel = new LinkedHashMap<String,List<SpikingMembrane>>();
		
		environment = new SpikingEnvironment(envLabel,this);
		secureAdd(environment);
		environment.setStepsTaken(0L);
		
		input = null;
		dictionary = new LinkedHashMap<String,Set<String>>();
		cloneMap = null;
		
		graph = new LinkedHashMap<Integer, Set<Integer>>();
		rgraph = new LinkedHashMap<Integer, Set<Integer>>(); 
		
		output = new HashSet<Integer>(); 
		
		showBinarySequence = false;
		showNaturalSequence = null;
		showSummatories = false;
		sequentialMode = 0;
		asynchMode = 0;
		asynchValidConfiguration = new HashMap<String, Long>();
		

	}

	public SpikingMembraneStructure(MembraneStructure membrane)
	{
		super();
				

		cellsById = new LinkedHashMap<Integer,SpikingMembrane>();
		cellsByLabel = new LinkedHashMap<String,List<SpikingMembrane>>();
		input = null;
		dictionary = null;
		cloneMap = new LinkedHashMap<Integer,Integer>();	// this is used to clone the structure mapping old to new id's
		
		graph = new LinkedHashMap<Integer, Set<Integer>>();
		rgraph = new LinkedHashMap<Integer, Set<Integer>>(); 
		
		output = new HashSet<Integer>(); 
		
		showBinarySequence = false;
		showNaturalSequence = null;
		showSummatories = false;
		sequentialMode = 0;
		asynchMode = 0;
		asynchValidConfiguration = new HashMap<String, Long>();
		
		Iterator<? extends Membrane>it;

		if (membrane instanceof SpikingMembraneStructure)
		{
			SpikingMembraneStructure sps = (SpikingMembraneStructure)membrane;
			environment = new SpikingEnvironment(sps.getEnvironmentLabel(),this);
			environment.getMultiSet().addAll(sps.getEnvironment());
			secureAdd(environment);							// this will add the environment in any case
			environment.setStepsTaken(sps.environment.getStepsTaken());
			
			this.showBinarySequence = sps.showBinarySequence;
			this.showNaturalSequence = sps.showNaturalSequence;
			this.showSummatories = sps.showSummatories;
			this.sequentialMode = sps.sequentialMode;
			this.asynchMode = sps.asynchMode;
			this.asynchValidConfiguration = sps.asynchValidConfiguration;
			
			it = membrane.getAllMembranes().iterator();

			while(it.hasNext())
			{
				Membrane m = it.next();
				
				if (m.getLabel().equals(environment.getLabel()))
				{
					if (membrane instanceof CellLikeSkinMembrane)
						throw new IllegalArgumentException("The environment label cannot be used as membrane label");
				
					if (m instanceof ChangeableMembrane)
						cloneMap.put(((ChangeableMembrane)m).getId(), environment.getId());
					else
						throw new IllegalArgumentException("Changeable Membranes are needed in otder to Map Ids");
				}	
				else
				{
					SpikingMembrane mem = new SpikingMembrane(m,this);
					add(mem);
					
					if (m instanceof ChangeableMembrane)
						cloneMap.put(((ChangeableMembrane)m).getId(), mem.getId());
					else
						throw new IllegalArgumentException("Changeable Membranes are needed in otder to Map Ids");
				
				}
				
			}
			
			input = ((sps.input == null) ? null: cloneMap.get(sps.input));	// with this, we get the right id
			
			if(input != null)	
				environment.setInputSequence(sps.environment.getInputSequence());
			
			// Now we clone the dictionary
		
			dictionary = sps.dictionary;			// As the dictionary itself is static we can assign it safely.
			
			// the output membranes are calculated as the predecessors to the environment
			// the output list can be automatically created as the edges are built
			// but for efficiency reasons we build also an output set
			// additionally for the output membranes we must clone their spike trains also
			// so we define the necessary variables first
			
			HashMap<Integer, ArrayList<Short>> spsBinarySpikeTrain = sps.environment.getBinarySpikeTrain();
			HashMap<Integer, ArrayList<Short>> thisBinarySpikeTrain = this.environment.getBinarySpikeTrain();
			HashMap<Integer, ArrayList<Integer>> spsNaturalSpikeTrain = sps.environment.getNaturalSpikeTrain();
			HashMap<Integer, ArrayList<Integer>> thisNaturalSpikeTrain = this.environment.getNaturalSpikeTrain();
			
			Iterator<Integer> keys = sps.graph.keySet().iterator();
			
			while(keys.hasNext())
			{
				Integer key = keys.next();
				
				Iterator<Integer> values = sps.graph.get(key).iterator();
				
				while(values.hasNext())
				{
					Integer value = values.next();
					
					int spsSourceId, spsTargetId, thisSourceId, thisTargetId;
					
					spsSourceId 		= key;
					spsTargetId 		= value;
				
					thisSourceId 		= cloneMap.get(spsSourceId);
					thisTargetId 		= cloneMap.get(spsTargetId);
					
					connect(thisSourceId, thisTargetId, false, false);
					// we don't need to build the dictionary as it can be assigned safely
					// we don't need to initialize the spike train as it's going to be cloned
					
					SpikingMembrane thisTarget = this.getCellById(thisTargetId);
					
					// if target is the environment then source is an output membrane and we have to clone its spike trains
					if(thisTarget.getLabel().equals(environment.getLabel()))
					{
						ArrayList<Short> binarySpikeTrainArray = (ArrayList<Short>) spsBinarySpikeTrain.get(spsSourceId).clone();
						thisBinarySpikeTrain.put(thisSourceId, binarySpikeTrainArray);
						ArrayList<Integer> naturalSpikeTrainArray = (ArrayList<Integer>) spsNaturalSpikeTrain.get(spsSourceId).clone();
						thisNaturalSpikeTrain.put(thisSourceId, naturalSpikeTrainArray);
						
						// we add the membrane to the output set
						output.add(thisSourceId);
					
					}
					
					
				}
			}
								

			
		}
		else
			throw new IllegalArgumentException("The membrane structure must be kinda Spiking one");
		
		cloneMap = null;	// this line is not strictly necessary as memory will be cleaned up.
				
	}

	protected int getNextId()
	{
		
		return cellsById.size();
	}
	

	public String getEnvironmentLabel() {
		return environment.getLabel();
	}
	
	public SpikingEnvironment getEnvironmentMembrane()
	{
		return environment;
	}
	
	@Override
	public Object clone()
	{
		SpikingMembraneStructure clone = new SpikingMembraneStructure(this);
		return clone;
	}
	
	@Override
	public Collection<? extends Membrane> getAllMembranes() {
		// TODO Auto-generated method stub
		return new SpikingMembraneCollection();
	}
	
	
	

	public SpikingMembrane getCellById(int id)
	{
		return cellsById.get(id);
	}
	
	
	public List<SpikingMembrane> getCellsByLabel(String label)
	{
		return cellsByLabel.get(label);
	}
	

	protected boolean renewLabel(SpikingMembrane m, String oldLabel, String newLabel)
	{
		
		List<SpikingMembrane> l = cellsByLabel.get(oldLabel);
		
		int i = 0;
		boolean stop = false;
		while(i < l.size() && !stop)
		{
			SpikingMembrane mem = (SpikingMembrane) l.get(i);
			
			if(mem.getId() == m.getId())
				stop = true;
			else
				i++;
			
		}
		
		if(stop)
		{
			l.remove(i);
			
			if(l.isEmpty())
				cellsByLabel.remove(oldLabel);
		
			if (!cellsByLabel.containsKey(newLabel))
			{
				l = new ArrayList<SpikingMembrane>();
				cellsByLabel.put(newLabel, l);
			}
			else
				l = cellsByLabel.get(newLabel);
			
			l.add(m);	// The membrane is always added to the list
			
			return true;
			
		}
		else
			return false;
	}
	
	
	private boolean secureAdd(SpikingMembrane arg0)
	{
		if (!cellsById.containsKey(arg0.getId()))
		{
			cellsById.put(arg0.getId(), arg0);
								
			String label = arg0.getLabel();
			
			List<SpikingMembrane>l;
			
			if (!cellsByLabel.containsKey(label))
			{
				l = new ArrayList<SpikingMembrane>();
				cellsByLabel.put(label, l);
			}
			else
				l = cellsByLabel.get(label);
			
			l.add(arg0);	// The membrane is always added to the list
			
			return true;
		}
		
		return false;
	}

	public boolean add(SpikingMembrane arg0)
	{
		// TODO Auto-generated method stub
		
		if (arg0.getLabel().equals(getEnvironmentLabel()))
			throw new IllegalArgumentException("Environment label");
		return secureAdd(arg0);
		
	}

	public InfiniteMultiSet<String> getEnvironment()
	{
		return (InfiniteMultiSet<String>)environment.getMultiSet();
	}
	
	public boolean connect(SpikingMembrane source, SpikingMembrane target)
	{
		return connect(source, target, true, true);
	}
	
	public boolean connect(String labelSource,String labelTarget)
	{
		return connect(labelSource,labelTarget,true,true);
	}
	public boolean connect(String labelSource,String labelTarget, boolean updateDictionary, boolean initializeSpikeTrain)
	{
		List<SpikingMembrane> sources = this.getCellsByLabel(labelSource);
		List<SpikingMembrane> targets = this.getCellsByLabel(labelTarget);
		if (sources==null || sources.isEmpty())
			throw new IllegalArgumentException("There's no source membranes with the specified label");
		if (targets==null || targets.isEmpty())
			throw new IllegalArgumentException("There's no target membranes with the specified label");
		
		boolean result = true;
		
		Iterator<SpikingMembrane> its = sources.iterator();
		
		while(result && its.hasNext())
		{
			SpikingMembrane s = (SpikingMembrane) its.next();
			
			Iterator<SpikingMembrane> itt = targets.iterator();
			
			while(result && itt.hasNext())
			{
				SpikingMembrane t = (SpikingMembrane) itt.next();
				
				result = connect(s,t,updateDictionary,initializeSpikeTrain);
			}
		}
		
		
		return result;
		
	}
	
	public boolean connect(Integer sourceId, Integer targetId, boolean updateDictionary, boolean initializeSpikeTrain)
	{
		SpikingMembrane source = this.getCellById(sourceId);
		SpikingMembrane target = this.getCellById(targetId);
		
		if (source==null)
			throw new IllegalArgumentException("The source membrane doesn't exist");
		if (target==null)
			throw new IllegalArgumentException("The target membrane doesn't exist");
		
		
		return connect(source,target,updateDictionary,initializeSpikeTrain);
	}
	
	
	public boolean connect(SpikingMembrane source, SpikingMembrane target, boolean updateDictionary, boolean initializeSpikeTrain)
	{
		boolean result = true;
		
		if(cellsById.containsKey(source.getId()) && cellsById.containsKey(target.getId()))
		{
		
				HashSet<Integer> set = null;
				
				if(graph.containsKey(source.getId()))
					set = (HashSet<Integer>) graph.get(source.getId());
				else
					set = new HashSet<Integer>();
				
				set.add(target.getId());
				graph.put(source.getId(), set);
				
				if(rgraph.containsKey(target.getId()))
					set = (HashSet<Integer>) rgraph.get(target.getId());
				else
					set = new HashSet<Integer>();
				
				set.add(source.getId());
				rgraph.put(target.getId(), set);
				
		}
		else
			result = false;
		
				
		// Now we update the dictionary but only when
		// - we are creating post sn p system creation synapses (budding / division) i. e. updateDictionary = true
		// - the connection went ok i. e. result = true
		// - we are not connecting to the environment as the environment has a special label not contained in the dictionary
		
		if(updateDictionary && result && !target.getLabel().equals(this.getEnvironmentLabel()))
		{
			String sourceLabel = source.getLabel();
			String targetLabel = target.getLabel();
			updateDictionary(sourceLabel,targetLabel);

		}
		
		// if we are connecting a membrane to the environment then we are setting an output membrane
		// so we have to mark it as output and initialize its spike trains
		
		if(result && target.getLabel().equals(this.getEnvironmentLabel()))
		{
			output.add(source.getId());
			
			if(initializeSpikeTrain)
				this.environment.initializeSpikeTrain(source);
		}
		
		return result;
	}
	
	public boolean updateDictionary(String sourceLabel, String targetLabel)
	{
		boolean result = true;
		
		HashSet<String> set = null;
		
		if(dictionary.containsKey(sourceLabel))
			set = (HashSet<String>) dictionary.get(sourceLabel);
		else
			set = new HashSet<String>();
		
		set.add(targetLabel);
		dictionary.put(sourceLabel, set);
			
		return result;
		
	}
	
	public boolean disconnect(SpikingMembrane source, SpikingMembrane target)
	{
		boolean result = true;
		
		if(cellsById.containsKey(source.getId()) && cellsById.containsKey(target.getId()))
		{
		
				HashSet<Integer> set = null;
				
				if(graph.containsKey(source.getId()))
					set = (HashSet<Integer>) graph.get(source.getId());
				else
					set = new HashSet<Integer>();
				
				if(set.contains(target.getId()))
					set.remove(target.getId());
				
				if(set.isEmpty())
					graph.remove(source.getId());
				else
					graph.put(source.getId(), set);
				
				
				if(rgraph.containsKey(target.getId()))
					set = (HashSet<Integer>) rgraph.get(target.getId());
				else
					set = new HashSet<Integer>();
				
				if(set.contains(source.getId()))
					set.remove(source.getId());
				
				if(set.isEmpty())
					rgraph.remove(target.getId());
				else
					rgraph.put(target.getId(), set);
				
		}
		else
			result = false;
		
		// if we are disconnecting a membrane from the environment we have to mark it as no output and clear its spike trains
		
		if(result && target.getLabel().equals(this.getEnvironmentLabel()))
		{
			output.remove(source.getId());
			this.environment.destroySpikeTrain(source);
		}
		
		return result;
	}
	
	public Map<String,Set<String>> getDictionary()
	{
		return dictionary;
		
	}
	
	public List<SpikingMembrane> getPredecessors(SpikingMembrane m)
	{

		List<SpikingMembrane> predecessors = new ArrayList<SpikingMembrane>();
		
		if(rgraph.containsKey(m.getId()))
		{
			Iterator<Integer>itedges = rgraph.get(m.getId()).iterator();
			
			while(itedges.hasNext())
			{
				Integer e = (Integer) itedges.next();
				
				SpikingMembrane s = this.getCellById(e);
		
				predecessors.add(s);			
			}
		}
		
				
		return predecessors;
	}
		
	
	public List<SpikingMembrane> getSuccessors(SpikingMembrane m)
	{

		
		List<SpikingMembrane> successors = new ArrayList<SpikingMembrane>();
		
		if(graph.containsKey(m.getId()))
		{
		
			Iterator<Integer>itedges = graph.get(m.getId()).iterator();
								
			while(itedges.hasNext())
			{
				Integer e = (Integer) itedges.next();
				
				SpikingMembrane s = this.getCellById(e);
				
				successors.add(s);			
			}
			
		}
		
		return successors;
	}
	
	public void setInputMembrane(String inputMembraneLabel,boolean check)
	{
		List<SpikingMembrane>l = this.getCellsByLabel(inputMembraneLabel);
		
		if (l==null || l.isEmpty())
			throw new IllegalArgumentException("The input membrane doesn't exist");
		setInputMembrane(l.get(0),check);
	}
		
	public void setInputMembrane(SpikingMembrane m, boolean check)
	{
		if(check)
			if(m == null || cellsById.containsKey(m.getId()) == false)
				throw new IllegalArgumentException("The membrane is not contained in the structure");
		
		this.input = m.getId();
		
		HashMap<Long,Long> inputSequence = new HashMap<Long,Long>();
		environment.setInputSequence(inputSequence);
				
	}
	
	public void setInputMembrane(SpikingMembrane m, HashMap<Long,Long> inputSequence, boolean check)
	{
		if(check)
			if(m == null || cellsById.containsKey(m.getId()) == false)
				throw new IllegalArgumentException("The membrane is not contained in the structure");
		
		this.input = m.getId();
		
		environment.setInputSequence(inputSequence);
		
		
	}
	
	public SpikingMembrane getInputMembrane()
	{
		return ((input == null) ? null : cellsById.get(input));
	}
		
	public void setOutputMembrane(String outputMembraneLabel,boolean check)
	{
		List<SpikingMembrane>l = this.getCellsByLabel(outputMembraneLabel);
		
		if (l==null || l.isEmpty())
			throw new IllegalArgumentException("The output membrane doesn't exist");
		
		
		setOutputMembranes(l,check);
		
	}
	
	public void setOutputMembranes(List<SpikingMembrane> o, boolean check)
	{
		if(check)
			if (o == null || cellsById.values().containsAll(o) == false)
				throw new IllegalArgumentException("The membranes are not contained in the structure");
			
		Iterator<SpikingMembrane> it = o.iterator();
		
		while(it.hasNext())
		{
			SpikingMembrane mem = (SpikingMembrane) it.next();
			connect(mem, environment, false, true);
			
			// we don't need to update the dictionary as we are setting output membranes
			// we have to initialize the spike train as we are setting output membranes
			
		}
						

	}
	
	public List<SpikingMembrane> getOutputMembranes()
	{
		return this.getPredecessors(environment);
	}
	
	public boolean isOutput(SpikingMembrane in)
	{
		boolean result = false;
		
		result = output.contains(in.getId());
		
		return result;
	}
	
	public boolean getShowBinarySequence()
	{
		return this.showBinarySequence;
	}
	
	public void setShowBinarySequence(boolean s)
	{
		this.showBinarySequence = s;
	}
	
	public ArrayList getShowNaturalSequence()
	{
		return this.showNaturalSequence;
	}
	
	public void setShowNaturalSequence(ArrayList s)
	{
		this.showNaturalSequence = s;
	}
	
	public boolean getShowSummatories()
	{
		return this.showSummatories;
	}
	
	public void setShowSummatories(boolean s)
	{
		this.showSummatories = s;
	}
	
	public int getSequentialMode()
	{
		return this.sequentialMode;
	}
	
	public void setSequentialMode(int i)
	{
		this.sequentialMode = i;
	}
	
	public int getAsynchMode()
	{
		return this.asynchMode;
	}
	
	public void setAsynchMode(int i)
	{
		this.asynchMode = i;
	}
	
	public Map<String, Long> getAsynchValidConfiguration()
	{
		return this.asynchValidConfiguration;
	}
	
	public void updateAsynchValidConfiguration(String label, long spikes)
	{
		this.asynchValidConfiguration.put(label, spikes);
	}

	@Override
	public String toString() {
		// TODO Auto-generated method stub
		String str="Cells: ";
		Iterator<? extends Membrane>it = getAllMembranes().iterator();
		while (it.hasNext())
		{
			str+=it.next().toString();
			if (it.hasNext())
				str+=", ";
		}
		
		str+='\n';
		
		str+="Arcs: ";
		
		str += '\n' + graph.toString();
				
		str+='\n';
		
		str+="Dictionary: ";
		str+='\n';
		Iterator<? extends String>itdict = dictionary.keySet().iterator();
		while (itdict.hasNext())
		{
			String key = itdict.next().toString(); 
			str+=key + ":" + dictionary.get(key).toString();
			if (itdict.hasNext())
				str+='\n';
		}
		
		str+='\n';
		
		str+="Input Membrane: ";
		str+= input == null ? "" : getCellById(input).getId();
		
		str+='\n';
		
		str+="Ouput Membranes: ";
		
		Iterator<? extends SpikingMembrane>it3 = this.getOutputMembranes().iterator();
		while (it3.hasNext())
		{
			str+= ((SpikingMembrane) it3.next()).getId();
			if (it3.hasNext())
				str+=", ";
		}
		
		return str;
	}
	
	


	class SpikingMembraneCollection implements Collection<SpikingMembrane>
	{
		
		public SpikingMembraneCollection() {
			super();
			// TODO Auto-generated constructor stub
		}

		@Override
		public boolean add(SpikingMembrane arg0) {
			throw new UnsupportedOperationException();
		}
		
		@Override
		public boolean addAll(Collection<? extends SpikingMembrane> arg0) {
			throw new UnsupportedOperationException();
		}

		
		@Override
		public void clear() {
			// TODO Auto-generated method stub
			throw new UnsupportedOperationException();
		}


		@Override
		public boolean contains(Object arg0) {
			// TODO Auto-generated method stub
			if (!(arg0 instanceof SpikingMembrane))
				return false;
			SpikingMembrane tlm = (SpikingMembrane)arg0;
			
			return cellsById.containsKey(tlm.getId());
		}

		public List<SpikingMembrane> getByLabel(Object arg0) {
			// TODO Auto-generated method stub
			if (!(arg0 instanceof String))
				return null;
			
			String label = (String)arg0;
			
			return cellsByLabel.get(label);
			
		}
		
		public SpikingMembrane getById(Object arg0) {
			// TODO Auto-generated method stub
			if (!(arg0 instanceof Integer))
				return null;
			
			Integer id = ((Integer)arg0).intValue();
			
			return cellsById.get(id);
			
		}

		
		@Override
		public boolean containsAll(Collection<?> arg0) {
			// TODO Auto-generated method stub
			Iterator<?>it = arg0.iterator();
			boolean contains=true;
			while(contains && it.hasNext())
				contains=contains(it.next());
			return contains;
		}


		@Override
		public boolean isEmpty() {
			// TODO Auto-generated method stub
			return cellsById.isEmpty();
		}



		@Override
		public Iterator<SpikingMembrane> iterator() {
			// TODO Auto-generated method stub
			return cellsById.values().iterator();
		}

		
		

		@Override
		public boolean remove(Object arg0) {
			// TODO Auto-generated method stub
			throw new UnsupportedOperationException();
		}



		@Override
		public boolean removeAll(Collection<?> arg0) {
			// TODO Auto-generated method stub
			throw new UnsupportedOperationException();
		}



		@Override
		public boolean retainAll(Collection<?> arg0) {
			// TODO Auto-generated method stub
			throw new UnsupportedOperationException();
		}



		@Override
		public int size() {
			// TODO Auto-generated method stub
			return cellsById.size();
		}



		@Override
		public Object[] toArray() {
			// TODO Auto-generated method stub
			throw new UnsupportedOperationException();
			
		}



		@Override
		public <T> T[] toArray(T[] arg0) {
			// TODO Auto-generated method stub
			throw new UnsupportedOperationException();
		}


	}

	@Override
	public Membrane getMembrane(int id) {
		// TODO Auto-generated method stub
		return cellsById.get(id);
	}
   
	

}
