/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */


package org.gcn.plinguacore.simulator.cellLike.stochastic;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.gcn.plinguacore.simulator.ISimulator;
import org.gcn.plinguacore.util.PlinguaCoreException;



/**
 * Clase StochasticTester

 *  @author Research Group on Natural Computing (http://www.gcn.us.es)
 *
 *
 * 
 * Esta clase proporciona funcionalidad para testear el simualdor estocástico con 
 * instancias del Discrete Stochastic Models Test Suite
 * (ver http://code.google.com/p/dsmts/)
 */


public class StochasticTester {

	private double end; // fin de cada simulación
	private int numberOfSteps;

	private int numberOfRuns; // número de simulaciones

	private Map<String, double[]> variables; // variables que representan las
	// especies del sistema

	private Map<String, double[]> means;
	private Map<String, double[]> standarDesviations;

	private Map<String, double[]> meansFromModel;
	private Map<String, double[]> sdFromModel;

	private ISimulator simulator; // Simulator to test


	/**
	 * @param end
	 * @param numberOfRuns
	 * @param simulator
	 */
	public StochasticTester(double end, int numberOfRuns, int numberOfSteps,
			ISimulator simulator) {
		super();

		this.end = end;

		this.numberOfRuns = numberOfRuns;
		this.simulator = simulator;
		variables = new TreeMap<String, double[]>();
		means = new TreeMap<String, double[]>();
		standarDesviations = new TreeMap<String, double[]>();
		meansFromModel = new TreeMap<String, double[]>();
		sdFromModel = new TreeMap<String, double[]>();
		this.numberOfSteps = numberOfSteps;
	}

	/**
	 * Performs the test, by running the simulator numberOfRuns times
	 */
	public void test(File referenceModel, File referenceModelSd) {

		// first, read the means and sd from the referenceModel
		BufferedReader bf;
		try {
			bf = new BufferedReader(new FileReader(referenceModel));
			String line = bf.readLine();
			String[] headers = line.split(",");
			double[][] values = new double[headers.length - 1][numberOfSteps + 1];
			line = bf.readLine();
			int j = 0;
			while (line != null && j <= numberOfSteps) {
				String[] aux = line.split(",");
				for (int i = 1; i < aux.length; i++)
					values[i - 1][j] = Double.parseDouble(aux[i]);
				j++;
				line = bf.readLine();
			}
			bf.close();

			for (int i = 0; i < values.length; i++)
				meansFromModel.put(headers[i + 1], values[i]);

			bf = new BufferedReader(new FileReader(referenceModelSd));
			line = bf.readLine();
			headers = line.split(",");
			values = new double[headers.length - 1][numberOfSteps + 1];
			line = bf.readLine();
			j = 0;
			while (line != null && j <= numberOfSteps) {
				String[] aux = line.split(",");
				for (int i = 1; i < aux.length; i++)
					values[i - 1][j] = Double.parseDouble(aux[i]);
				j++;
				line = bf.readLine();
			}
			bf.close();

			for (int i = 0; i < values.length; i++)
				sdFromModel.put(headers[i + 1], values[i]);

		} catch (FileNotFoundException e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		for (int i = 0; i < numberOfRuns; i++) {
			System.out.println("Starting simulation " + i);
			simulator.reset();
			((StochasticSimulator) simulator).setTimeOut(end);
			((StochasticSimulator) simulator).setNumberOfSteps(numberOfSteps);
			try {
				simulator.run();
			} catch (PlinguaCoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

			variables = ((StochasticSimulator) simulator).getResults();

			// Hemos hecho hecho una de las numberOfRuns simulaciones
			// El atributo "variables" contiene los valores de cada
			// variable para t= 0 ... end
			// ahora actualizamos los valores de la media y de las desviaciones
			// típicas
			Iterator<String> it = variables.keySet().iterator();
			while (it.hasNext()) {
				String var = it.next();
				double[] values = variables.get(var);
				double[] meanForVar = means.get(var);
				double[] sdForVar = standarDesviations.get(var);
				if (meanForVar == null)
					// primera vez que aparece var
					meanForVar = new double[numberOfSteps + 1];

				if (sdForVar == null)
					// idem
					sdForVar = new double[numberOfSteps + 1];

				double[] meanForVarModel = meansFromModel.get(var);

				if (meanForVarModel != null)
					for (int j = 0; j < meanForVar.length; j++) {
						meanForVar[j] += values[j];
						sdForVar[j] += (values[j] - meanForVarModel[j])
								* (values[j] - meanForVarModel[j]);
					}

				means.put(var, meanForVar);
				standarDesviations.put(var, sdForVar);
			}

		}

		// Hemos terminado con las numberOfRuns simulaciones
		// Cerramos el cálculo de todas las medias y desviaciones típicas

		Iterator<String> it = means.keySet().iterator();
		while (it.hasNext()) {
			String var = it.next();
			double[] meanForVar = means.get(var);
			double[] meanForVarModel = meansFromModel.get(var);
			double[] sdForVarModel = sdFromModel.get(var);
			double[] sdForVar = standarDesviations.get(var);

			if (sdForVar != null && meanForVarModel != null
					&& sdForVarModel != null) {
				for (int i = 0; i < meanForVar.length; i++) {
					meanForVar[i] = meanForVar[i] / numberOfRuns;

					sdForVar[i] = sdForVar[i] / numberOfRuns;
				}

				for (int i = 0; i < meanForVar.length; i++) {
					meanForVar[i] = Math.sqrt(numberOfRuns)
							* (meanForVar[i] - meanForVarModel[i])
							/ sdForVarModel[i];
					sdForVar[i] = Math.sqrt((double) numberOfRuns / 2)
							* (sdForVar[i]
									/ (sdForVarModel[i] * sdForVarModel[i]) - 1);
				}

				means.put(var, meanForVar);
				standarDesviations.put(var, sdForVar);

			}

		}

	}

	public void testResults() {
		int meanFails = 0;
		int sdFails = 0;
		Iterator<String> it = means.keySet().iterator();
		while (it.hasNext()) {
			String var = it.next();
			System.out.println(var);
			double[] values = means.get(var);
			for (int i = 0; i < values.length; i++) {
				boolean fails = (Math.abs(values[i]) > 3);
				System.out.println(values[i] + " fails=" + fails);
				if (fails)
					meanFails++;
			}

			values = standarDesviations.get(var);
			if (values != null)
				for (int i = 0; i < values.length; i++) {
					boolean fails = (Math.abs(values[i]) > 5);
					System.out.println(values[i] + " fails=" + fails);
					if (fails)
						sdFails++;

				}
		}
		System.out.println(meanFails + " mean fails");
		System.out.println(sdFails + " sd fails");
	}

	public Map<String, double[]> getMeans() {
		return means;
	}

	public Map<String, double[]> getStandarDesviations() {
		return standarDesviations;
	}

	private String writeArray(double[] array) {
		String res = "[ ";
		for (int i = 0; i < array.length; i++)
			res += array[i] + " ";

		return res;
	}

	// private void write(String file, Map<String, double[]> toWrite) {
	// try {
	// PrintStream p = new PrintStream(file);
	// double T = start;
	// for (int j=0; j<numberOfSteps; j++) {
	// if (j==0) {
	// // imprimimos las cabeceras del archivo
	// p.print("#time");
	// Iterator<String> it = toWrite.keySet().iterator();
	// while (it.hasNext()) {
	// String var = it.next();
	// p.print(","+var);
	// }
	// }
	//				
	// // imprimimos los valores para T
	// p.println();
	// p.format("%.2f", T);
	// Iterator<String> it = toWrite.keySet().iterator();
	// while (it.hasNext()) {
	// String var = it.next();
	// double [] values = toWrite.get(var);
	// p.print(","+values[j]);
	// }
	// T = T + stepSize;
	// }
	//			
	// } catch (FileNotFoundException e) {
	// // TODO Auto-generated catch block
	// e.printStackTrace();
	// }
	// }

	// public void writeStandarDesviations(String name) {
	// write(name+"-mean.csv",means);
	// }
	//	
	// public void writeMeans(String name) {
	// write(name+"-sd.csv",standarDesviations);
	// }

}
