/* 
 * pLinguaCore: A JAVA library for Membrane Computing
 *              http://www.p-lingua.org
 *
 * Copyright (C) 2009  Research Group on Natural Computing
 *                     http://www.gcn.us.es
 *                      
 * This file is part of pLinguaCore.
 *
 * pLinguaCore is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * pLinguaCore is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with pLinguaCore.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.gcn.plinguacore.parser.output.binary;

import java.io.DataOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Writer;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.gcn.plinguacore.parser.output.OutputParser;
import org.gcn.plinguacore.util.HashMultiSet;
import org.gcn.plinguacore.util.MultiSet;
import org.gcn.plinguacore.util.psystem.AlphabetObject;
import org.gcn.plinguacore.util.psystem.Psystem;
import org.gcn.plinguacore.util.psystem.cellLike.membrane.CellLikeMembrane;
import org.gcn.plinguacore.util.psystem.cellLike.membrane.CellLikeNoSkinMembrane;
import org.gcn.plinguacore.util.psystem.cellLike.membrane.CellLikeSkinMembrane;
import org.gcn.plinguacore.util.psystem.membrane.Membrane;
import org.gcn.plinguacore.util.psystem.rule.IRule;

abstract class AbstractBinaryOutputParser extends OutputParser {

	protected static final byte ACTIVE_MEMBRANES_ID=0x11;
	protected static final byte PROBABILISTIC_ID=0x20;
	
	
	private DataOutputStream stream = null;
	private Psystem psystem = null;
	private Map<String, Integer> alphabet = null;
	Map<String, Integer> labels = null;
	private Map<Integer, MultiSet<String>> multiSets = null;
	
	public AbstractBinaryOutputParser()
	{
		multiSets = new LinkedHashMap<Integer, MultiSet<String>>();
		alphabet = new LinkedHashMap<String, Integer>();
		labels = new LinkedHashMap<String, Integer>();
	}
	
	protected void writeHead(byte id) throws IOException {
		stream.writeByte(0xAF);
		stream.writeByte(0x12);
		stream.writeByte(0xFA);
		stream.writeByte(id);
	}
	
	protected abstract void writeHead() throws IOException;
	
	protected void writeAlphabet() throws IOException {
		alphabet.clear();
		int n = psystem.getAlphabet().size();
		stream.writeChar(n+1);
		alphabet.put("#", 0);
		stream.writeBytes("#");
		stream.writeByte(0);
		Iterator<AlphabetObject> it = psystem.getAlphabet().iterator();
		int i = 1;
		while (it.hasNext()) {
			String obj = it.next().toString();
			if (!alphabet.containsKey(obj)) {
				alphabet.put(obj, i);
				i++;
				stream.writeBytes(obj);
				stream.writeByte(0);
			}
		}

	}

	
	protected void writeLabels() throws IOException {
		labels.clear();
		Iterator<? extends Membrane> it = getPsystem().getMembraneStructure().getAllMembranes().iterator();
		int i = 0;
		while (it.hasNext()) {
			String label = it.next().getLabelObj().toString();
			if (!labels.containsKey(label)) {
				labels.put(label, i);
				i++;
			}
		}
		getStream().writeChar(labels.size());
		Iterator<String> it1 = labels.keySet().iterator();
		while (it1.hasNext()) {
			getStream().writeBytes(it1.next());
			getStream().writeByte(0);
		}

	}
	
	protected void writeMembranes() throws IOException {
		int n = getPsystem().getMembraneStructure().getAllMembranes().size();
		multiSets.clear();
		getStream().writeChar(n);
		CellLikeSkinMembrane skin = (CellLikeSkinMembrane)getPsystem().getMembraneStructure();
			
		
		if (skin != null)
			writeMembrane(skin);

	}
	protected void writeId(int id, int size) throws IOException {
		if (size > 0xFF)
			getStream().writeChar(id);
		else
			getStream().writeByte(id);
	}
	
	protected void writeCharge(byte charge) throws IOException
	{
		if (charge == 0)
			getStream().writeByte(0);
		else if (charge > 0)
			getStream().writeByte(1);
		else
			getStream().writeByte(2);

	}
	
	protected void writeObject(String obj) throws IOException
	{
		int id = getAlphabet().get(obj);
		writeObjectId(id);
	}
	

	protected void writeObjects(MultiSet<String> objs,boolean withMultiplicity,boolean forced) throws IOException
	{
		if (objs.size()==0 && forced)
		{
			writeObject("#");
			if (withMultiplicity)
				getStream().writeChar(1);
		}
		else
		{
			Iterator<String> it=objs.entrySet().iterator();
			while(it.hasNext())
			{
				String obj = it.next();
				writeObject(obj);
				if (withMultiplicity)
					getStream().writeChar((int)objs.count(obj));
			}
		}
	}
	
	
	
	protected void writeRuleHead(IRule r) throws IOException
	{
		writeLabelId(labels.get(r.getLeftHandRule().getOuterRuleMembrane()
				.getLabel()));
		writeCharge(r.getLeftHandRule().getOuterRuleMembrane().getCharge());
	}

	
	protected void writeMembraneId(int id) throws IOException {
		writeId(id, getPsystem().getMembraneStructure().getAllMembranes().size());
	}

	protected void writeLabelId(int id) throws IOException {
		writeId(id, labels.size());
	}

	protected void writeObjectId(int id) throws IOException {
		writeId(id, getAlphabet().size());
	}
	
	private void writeMembrane(CellLikeMembrane membrane) throws IOException {
		if (membrane instanceof CellLikeSkinMembrane)
			writeMembraneId(0);
		else
			writeMembraneId(((CellLikeNoSkinMembrane) membrane)
					.getParentMembrane().getId());
		
		String labelWithEnvironment = membrane.getLabelObj().toString();
		String label = membrane.getLabel();
		MultiSet<String> initMs = new HashMultiSet<String>();
		initMs.addAll(membrane.getMultiSet());
		
		if (getPsystem().getInitialMultiSets().get(label)!=null && 
				!getPsystem().getInitialMultiSets().get(label).isEmpty())
			initMs.addAll(getPsystem().getInitialMultiSets().get(label));
	
		if (!initMs.isEmpty())
			multiSets.put(membrane.getId(), initMs);
		
		writeLabelId(labels.get(labelWithEnvironment));
		if (membrane.getCharge() == 0)
			getStream().writeByte(0);
		else if (membrane.getCharge() > 0)
			getStream().writeByte(1);
		else
			getStream().writeByte(2);
		Iterator<CellLikeNoSkinMembrane> it = membrane.getChildMembranes()
				.iterator();
		while (it.hasNext())
			writeMembrane(it.next());

	}

	
	protected void writeMultiSets() throws IOException {
		getStream().writeChar(multiSets.size());
		Iterator<Integer> it = multiSets.keySet().iterator();
		while (it.hasNext()) {
			int id = it.next();
			writeMembraneId(id);
			MultiSet<String> ms = multiSets.get(id);
			getStream().writeChar(ms.entrySet().size());
			Iterator<String> it1 = ms.entrySet().iterator();
			while (it1.hasNext()) {
				String obj = it1.next();
				long mul = ms.count(obj);
				int objId = getAlphabet().get(obj);
				writeObjectId(objId);
				getStream().writeChar((int)mul);
			}
		}
	}

	
	
	
	
	protected final Map<String, Integer> getAlphabet() {
		return alphabet;
	}

	protected final void setAlphabet(Map<String, Integer> alphabet) {
		this.alphabet = alphabet;
	}

	protected final DataOutputStream getStream() {
		return stream;
	}

	protected final void setStream(DataOutputStream stream) {
		this.stream = stream;
	}

	protected final Psystem getPsystem() {
		return psystem;
	}

	protected final void setPsystem(Psystem psystem) {
		this.psystem = psystem;
	}

	@Override
	public final boolean parse(Psystem psystem, Writer stream) {
		// TODO Auto-generated method stub
		throw new UnsupportedOperationException();
	}
	
	protected abstract void writeRules() throws IOException;
	
	@Override
	public boolean parse(Psystem psystem, OutputStream stream) {
		
		try {
			
			setPsystem(psystem);
			setStream(new DataOutputStream(stream));
			writeHead();
			writeAlphabet();
			writeLabels();
			writeMembranes();
			writeMultiSets();
			writeRules();
			getStream().close();

		} catch (IOException e) {

			return false;
		}
		return true;

	}

}
